"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const receipt_rule_1 = require("./receipt-rule");
const ses_generated_1 = require("./ses.generated");
/**
 * A new or imported receipt rule set.
 */
class ReceiptRuleSetBase extends core_1.Resource {
    /**
     * Adds a new receipt rule in this rule set. The new rule is added after
     * the last added rule unless `after` is specified.
     */
    addRule(id, options) {
        this.lastAddedRule = new receipt_rule_1.ReceiptRule(this, id, {
            after: this.lastAddedRule ? this.lastAddedRule : undefined,
            ruleSet: this,
            ...options,
        });
        return this.lastAddedRule;
    }
    /**
     * Adds a drop spam rule
     */
    addDropSpamRule() {
        const dropSpam = new receipt_rule_1.DropSpamReceiptRule(this, 'DropSpam', {
            ruleSet: this,
        });
        this.lastAddedRule = dropSpam.rule;
    }
}
/**
 * A new receipt rule set.
 */
class ReceiptRuleSet extends ReceiptRuleSetBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.receiptRuleSetName,
        });
        const resource = new ses_generated_1.CfnReceiptRuleSet(this, 'Resource', {
            ruleSetName: this.physicalName,
        });
        this.receiptRuleSetName = resource.ref;
        if (props) {
            const rules = props.rules || [];
            rules.forEach((ruleOption, idx) => this.addRule(`Rule${idx}`, ruleOption));
            if (props.dropSpam) {
                this.addDropSpamRule();
            }
        }
    }
    /**
     * Import an exported receipt rule set.
     */
    static fromReceiptRuleSetName(scope, id, receiptRuleSetName) {
        class Import extends ReceiptRuleSetBase {
            constructor() {
                super(...arguments);
                this.receiptRuleSetName = receiptRuleSetName;
            }
        }
        return new Import(scope, id);
    }
}
exports.ReceiptRuleSet = ReceiptRuleSet;
//# sourceMappingURL=data:application/json;base64,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