"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Saves the received message to an Amazon S3 bucket and, optionally, publishes
 * a notification to Amazon SNS.
 */
class S3 {
    constructor(props) {
        this.props = props;
    }
    bind(rule) {
        // Allow SES to write to S3 bucket
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-s3
        const keyPattern = this.props.objectKeyPrefix || '';
        const s3Statement = new iam.PolicyStatement({
            actions: ['s3:PutObject'],
            principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
            resources: [this.props.bucket.arnForObjects(`${keyPattern}*`)],
            conditions: {
                StringEquals: {
                    'aws:Referer': cdk.Aws.ACCOUNT_ID,
                },
            },
        });
        this.props.bucket.addToResourcePolicy(s3Statement);
        const policy = this.props.bucket.node.tryFindChild('Policy');
        if (policy) { // The bucket could be imported
            rule.node.addDependency(policy);
        }
        else {
            rule.node.addWarning('This rule is using a S3 action with an imported bucket. Ensure permission is given to SES to write to that bucket.');
        }
        // Allow SES to use KMS master key
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-kms
        if (this.props.kmsKey && !/alias\/aws\/ses$/.test(this.props.kmsKey.keyArn)) {
            const kmsStatement = new iam.PolicyStatement({
                actions: ['km:Encrypt', 'kms:GenerateDataKey'],
                principals: [new iam.ServicePrincipal('ses.amazonaws.com')],
                resources: ['*'],
                conditions: {
                    Null: {
                        'kms:EncryptionContext:aws:ses:rule-name': 'false',
                        'kms:EncryptionContext:aws:ses:message-id': 'false',
                    },
                    StringEquals: {
                        'kms:EncryptionContext:aws:ses:source-account': cdk.Aws.ACCOUNT_ID,
                    },
                },
            });
            this.props.kmsKey.addToResourcePolicy(kmsStatement);
        }
        return {
            s3Action: {
                bucketName: this.props.bucket.bucketName,
                kmsKeyArn: this.props.kmsKey ? this.props.kmsKey.keyArn : undefined,
                objectKeyPrefix: this.props.objectKeyPrefix,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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