"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The type of invocation to use for a Lambda Action.
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * The function will be invoked asynchronously.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * The function will be invoked sychronously. Use RequestResponse only when
     * you want to make a mail flow decision, such as whether to stop the receipt
     * rule or the receipt rule set.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
/**
 * Calls an AWS Lambda function, and optionally, publishes a notification to
 * Amazon SNS.
 */
class Lambda {
    constructor(props) {
        this.props = props;
    }
    bind(rule) {
        // Allow SES to invoke Lambda function
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-lambda
        const permissionId = 'AllowSes';
        if (!this.props.function.permissionsNode.tryFindChild(permissionId)) {
            this.props.function.addPermission(permissionId, {
                action: 'lambda:InvokeFunction',
                principal: new iam.ServicePrincipal('ses.amazonaws.com'),
                sourceAccount: cdk.Aws.ACCOUNT_ID,
            });
        }
        // Ensure permission is deployed before rule
        const permission = this.props.function.permissionsNode.tryFindChild(permissionId);
        if (permission) { // The Lambda could be imported
            rule.node.addDependency(permission);
        }
        else {
            // tslint:disable-next-line:max-line-length
            rule.node.addWarning('This rule is using a Lambda action with an imported function. Ensure permission is given to SES to invoke that function.');
        }
        return {
            lambdaAction: {
                functionArn: this.props.function.functionArn,
                invocationType: this.props.invocationType,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.Lambda = Lambda;
//# sourceMappingURL=data:application/json;base64,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