"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const alias_target_instance_1 = require("./alias-target-instance");
const cname_instance_1 = require("./cname-instance");
const ip_instance_1 = require("./ip-instance");
const namespace_1 = require("./namespace");
const non_ip_instance_1 = require("./non-ip-instance");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
class ServiceBase extends core_1.Resource {
}
/**
 * Define a CloudMap Service
 */
class Service extends ServiceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const namespaceType = props.namespace.type;
        // Validations
        if (namespaceType === namespace_1.NamespaceType.HTTP && (props.routingPolicy || props.dnsRecordType)) {
            throw new Error('Cannot specify `routingPolicy` or `dnsRecord` when using an HTTP namespace.');
        }
        if (props.healthCheck && props.customHealthCheck) {
            throw new Error('Cannot specify both `healthCheckConfig` and `healthCheckCustomConfig`.');
        }
        if (namespaceType === namespace_1.NamespaceType.DNS_PRIVATE && props.healthCheck) {
            throw new Error('Cannot specify `healthCheckConfig` for a Private DNS namespace.');
        }
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.dnsRecordType === DnsRecordType.CNAME) {
            throw new Error('Cannot use `CNAME` record when routing policy is `Multivalue`.');
        }
        // Additional validation for eventual attachment of LBs
        // The same validation happens later on during the actual attachment
        // of LBs, but we need the property for the correct configuration of
        // routingPolicy anyway, so might as well do the validation as well.
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.loadBalancer) {
            throw new Error('Cannot register loadbalancers when routing policy is `Multivalue`.');
        }
        if (props.healthCheck
            && props.healthCheck.type === HealthCheckType.TCP
            && props.healthCheck.resourcePath) {
            throw new Error('Cannot specify `resourcePath` when using a `TCP` health check.');
        }
        // Set defaults where necessary
        const routingPolicy = (props.dnsRecordType === DnsRecordType.CNAME) || props.loadBalancer
            ? RoutingPolicy.WEIGHTED
            : RoutingPolicy.MULTIVALUE;
        const dnsRecordType = props.dnsRecordType || DnsRecordType.A;
        if (props.loadBalancer
            && (!(dnsRecordType === DnsRecordType.A
                || dnsRecordType === DnsRecordType.AAAA
                || dnsRecordType === DnsRecordType.A_AAAA))) {
            throw new Error('Must support `A` or `AAAA` records to register loadbalancers.');
        }
        const dnsConfig = props.namespace.type === namespace_1.NamespaceType.HTTP
            ? undefined
            : {
                dnsRecords: renderDnsRecords(dnsRecordType, props.dnsTtl),
                namespaceId: props.namespace.namespaceId,
                routingPolicy,
            };
        const healthCheckConfigDefaults = {
            type: HealthCheckType.HTTP,
            failureThreshold: 1,
            resourcePath: props.healthCheck && props.healthCheck.type !== HealthCheckType.TCP
                ? '/'
                : undefined,
        };
        const healthCheckConfig = props.healthCheck && { ...healthCheckConfigDefaults, ...props.healthCheck };
        const healthCheckCustomConfig = props.customHealthCheck;
        // Create service
        const service = new servicediscovery_generated_1.CfnService(this, 'Resource', {
            name: props.name,
            description: props.description,
            dnsConfig,
            healthCheckConfig,
            healthCheckCustomConfig,
            namespaceId: props.namespace.namespaceId,
        });
        this.serviceName = service.attrName;
        this.serviceArn = service.attrArn;
        this.serviceId = service.attrId;
        this.namespace = props.namespace;
        this.dnsRecordType = dnsRecordType;
        this.routingPolicy = routingPolicy;
    }
    static fromServiceAttributes(scope, id, attrs) {
        class Import extends ServiceBase {
            constructor() {
                super(...arguments);
                this.namespace = attrs.namespace;
                this.serviceId = attrs.serviceId;
                this.serviceArn = attrs.serviceArn;
                this.dnsRecordType = attrs.dnsRecordType;
                this.routingPolicy = attrs.routingPolicy;
                this.serviceName = attrs.serviceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Registers an ELB as a new instance with unique name instanceId in this service.
     */
    registerLoadBalancer(id, loadBalancer, customAttributes) {
        return new alias_target_instance_1.AliasTargetInstance(this, id, {
            service: this,
            dnsName: loadBalancer.loadBalancerDnsName,
            customAttributes,
        });
    }
    /**
     * Registers a resource that is accessible using values other than an IP address or a domain name (CNAME).
     */
    registerNonIpInstance(id, props) {
        return new non_ip_instance_1.NonIpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using an IP address.
     */
    registerIpInstance(id, props) {
        return new ip_instance_1.IpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using a CNAME.
     */
    registerCnameInstance(id, props) {
        return new cname_instance_1.CnameInstance(this, id, {
            service: this,
            ...props,
        });
    }
}
exports.Service = Service;
function renderDnsRecords(dnsRecordType, dnsTtl = core_1.Duration.minutes(1)) {
    const ttl = dnsTtl.toSeconds();
    if (dnsRecordType === DnsRecordType.A_AAAA) {
        return [{
                type: DnsRecordType.A,
                ttl,
            }, {
                type: DnsRecordType.AAAA,
                ttl,
            }];
    }
    else {
        return [{ type: dnsRecordType, ttl }];
    }
}
var DnsRecordType;
(function (DnsRecordType) {
    /**
     * An A record
     */
    DnsRecordType["A"] = "A";
    /**
     * An AAAA record
     */
    DnsRecordType["AAAA"] = "AAAA";
    /**
     * Both an A and AAAA record
     */
    DnsRecordType["A_AAAA"] = "A, AAAA";
    /**
     * A Srv record
     */
    DnsRecordType["SRV"] = "SRV";
    /**
     * A CNAME record
     */
    DnsRecordType["CNAME"] = "CNAME";
})(DnsRecordType = exports.DnsRecordType || (exports.DnsRecordType = {}));
var RoutingPolicy;
(function (RoutingPolicy) {
    /**
     * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
     * registered using the same service.
     */
    RoutingPolicy["WEIGHTED"] = "WEIGHTED";
    /**
     * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable value
     * for up to eight instances.
     */
    RoutingPolicy["MULTIVALUE"] = "MULTIVALUE";
})(RoutingPolicy = exports.RoutingPolicy || (exports.RoutingPolicy = {}));
var HealthCheckType;
(function (HealthCheckType) {
    /**
     * Route 53 tries to establish a TCP connection. If successful, Route 53 submits an HTTP request and waits for an HTTP
     * status code of 200 or greater and less than 400.
     */
    HealthCheckType["HTTP"] = "HTTP";
    /**
     * Route 53 tries to establish a TCP connection. If successful, Route 53 submits an HTTPS request and waits for an
     * HTTP status code of 200 or greater and less than 400.  If you specify HTTPS for the value of Type, the endpoint
     * must support TLS v1.0 or later.
     */
    HealthCheckType["HTTPS"] = "HTTPS";
    /**
     * Route 53 tries to establish a TCP connection.
     * If you specify TCP for Type, don't specify a value for ResourcePath.
     */
    HealthCheckType["TCP"] = "TCP";
})(HealthCheckType = exports.HealthCheckType || (exports.HealthCheckType = {}));
//# sourceMappingURL=data:application/json;base64,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