"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const destination_1 = require("../destination");
const notifications_resource_handler_1 = require("./notifications-resource-handler");
/**
 * A custom CloudFormation resource that updates bucket notifications for a
 * bucket. The reason we need it is because the AWS::S3::Bucket notification
 * configuration is defined on the bucket itself, which makes it impossible to
 * provision notifications at the same time as the target (since
 * PutBucketNotifications validates the targets).
 *
 * Since only a single BucketNotifications resource is allowed for each Bucket,
 * this construct is not exported in the public API of this module. Instead, it
 * is created just-in-time by `s3.Bucket.onEvent`, so a 1:1 relationship is
 * ensured.
 *
 * @see
 * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-notificationconfig.html
 */
class BucketNotifications extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaNotifications = new Array();
        this.queueNotifications = new Array();
        this.topicNotifications = new Array();
        this.bucket = props.bucket;
    }
    /**
     * Adds a notification subscription for this bucket.
     * If this is the first notification, a BucketNotification resource is added to the stack.
     *
     * @param event The type of event
     * @param target The target construct
     * @param filters A set of S3 key filters
     */
    addNotification(event, target, ...filters) {
        const resource = this.createResourceOnce();
        // resolve target. this also provides an opportunity for the target to e.g. update
        // policies to allow this notification to happen.
        const targetProps = target.bind(this, this.bucket);
        const commonConfig = {
            Events: [event],
            Filter: renderFilters(filters),
        };
        // if the target specifies any dependencies, add them to the custom resource.
        // for example, the SNS topic policy must be created /before/ the notification resource.
        // otherwise, S3 won't be able to confirm the subscription.
        if (targetProps.dependencies) {
            resource.node.addDependency(...targetProps.dependencies);
        }
        // based on the target type, add the the correct configurations array
        switch (targetProps.type) {
            case destination_1.BucketNotificationDestinationType.LAMBDA:
                this.lambdaNotifications.push({ ...commonConfig, LambdaFunctionArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.QUEUE:
                this.queueNotifications.push({ ...commonConfig, QueueArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.TOPIC:
                this.topicNotifications.push({ ...commonConfig, TopicArn: targetProps.arn });
                break;
            default:
                throw new Error('Unsupported notification target type:' + destination_1.BucketNotificationDestinationType[targetProps.type]);
        }
    }
    renderNotificationConfiguration() {
        return {
            LambdaFunctionConfigurations: this.lambdaNotifications.length > 0 ? this.lambdaNotifications : undefined,
            QueueConfigurations: this.queueNotifications.length > 0 ? this.queueNotifications : undefined,
            TopicConfigurations: this.topicNotifications.length > 0 ? this.topicNotifications : undefined,
        };
    }
    /**
     * Defines the bucket notifications resources in the stack only once.
     * This is called lazily as we add notifications, so that if notifications are not added,
     * there is no notifications resource.
     */
    createResourceOnce() {
        if (!this.resource) {
            const handlerArn = notifications_resource_handler_1.NotificationsResourceHandler.singleton(this);
            this.resource = new cdk.CfnResource(this, 'Resource', {
                type: 'Custom::S3BucketNotifications',
                properties: {
                    ServiceToken: handlerArn,
                    BucketName: this.bucket.bucketName,
                    NotificationConfiguration: cdk.Lazy.anyValue({ produce: () => this.renderNotificationConfiguration() }),
                },
            });
        }
        return this.resource;
    }
}
exports.BucketNotifications = BucketNotifications;
function renderFilters(filters) {
    if (!filters || filters.length === 0) {
        return undefined;
    }
    const renderedRules = new Array();
    let hasPrefix = false;
    let hasSuffix = false;
    for (const rule of filters) {
        if (!rule.suffix && !rule.prefix) {
            throw new Error('NotificationKeyFilter must specify `prefix` and/or `suffix`');
        }
        if (rule.suffix) {
            if (hasSuffix) {
                throw new Error('Cannot specify more than one suffix rule in a filter.');
            }
            renderedRules.push({ Name: 'suffix', Value: rule.suffix });
            hasSuffix = true;
        }
        if (rule.prefix) {
            if (hasPrefix) {
                throw new Error('Cannot specify more than one prefix rule in a filter.');
            }
            renderedRules.push({ Name: 'prefix', Value: rule.prefix });
            hasPrefix = true;
        }
    }
    return {
        Key: {
            FilterRules: renderedRules,
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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