"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
/**
 * Use an SQS queue as a bucket notification destination
 */
class SqsDestination {
    constructor(queue) {
        this.queue = queue;
    }
    /**
     * Allows using SQS queues as destinations for bucket notifications.
     * Use `bucket.onEvent(event, queue)` to subscribe.
     */
    bind(_scope, bucket) {
        this.queue.grantSendMessages(new iam.ServicePrincipal('s3.amazonaws.com', {
            conditions: {
                ArnLike: { 'aws:SourceArn': bucket.bucketArn },
            },
        }));
        // if this queue is encrypted, we need to allow S3 to read messages since that's how
        // it verifies that the notification destination configuration is valid.
        if (this.queue.encryptionMasterKey) {
            this.queue.encryptionMasterKey.addToResourcePolicy(new iam.PolicyStatement({
                principals: [new iam.ServicePrincipal('s3.amazonaws.com')],
                actions: ['kms:GenerateDataKey*', 'kms:Decrypt'],
                resources: ['*'],
            }), /* allowNoOp */ false);
        }
        return {
            arn: this.queue.queueArn,
            type: s3.BucketNotificationDestinationType.QUEUE,
            dependencies: [this.queue],
        };
    }
}
exports.SqsDestination = SqsDestination;
//# sourceMappingURL=data:application/json;base64,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