"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assets = require("../../assets"); // Automatically re-written from '@aws-cdk/assets'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const fs = require("fs");
const path = require("path");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * An asset represents a local file or directory, which is automatically uploaded to S3
 * and then can be referenced within a CDK application.
 */
class Asset extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // stage the asset source (conditionally).
        const staging = new assets.Staging(this, 'Stage', {
            ...props,
            sourcePath: path.resolve(props.path),
        });
        this.sourceHash = props.sourceHash || staging.sourceHash;
        this.assetPath = staging.stagedPath;
        const packaging = determinePackaging(staging.sourcePath);
        // sets isZipArchive based on the type of packaging and file extension
        this.isZipArchive = packaging === cdk.FileAssetPackaging.ZIP_DIRECTORY
            ? true
            : ARCHIVE_EXTENSIONS.some(ext => staging.sourcePath.toLowerCase().endsWith(ext));
        const stack = cdk.Stack.of(this);
        const location = stack.synthesizer.addFileAsset({
            packaging,
            sourceHash: this.sourceHash,
            fileName: staging.stagedPath,
        });
        this.s3BucketName = location.bucketName;
        this.s3ObjectKey = location.objectKey;
        this.s3ObjectUrl = location.s3ObjectUrl;
        this.httpUrl = location.httpUrl;
        this.s3Url = location.httpUrl; // for backwards compatibility
        this.bucket = s3.Bucket.fromBucketName(this, 'AssetBucket', this.s3BucketName);
        for (const reader of (props.readers || [])) {
            this.grantRead(reader);
        }
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/aws/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     * (e.g. "Code" for AWS::Lambda::Function)
     */
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * Grants read permissions to the principal on the assets bucket.
     */
    grantRead(grantee) {
        // we give permissions on all files in the bucket since we don't want to
        // accidentally revoke permission on old versions when deploying a new
        // version (for example, when using Lambda traffic shifting).
        this.bucket.grantRead(grantee);
    }
}
exports.Asset = Asset;
function determinePackaging(assetPath) {
    if (!fs.existsSync(assetPath)) {
        throw new Error(`Cannot find asset at ${assetPath}`);
    }
    if (fs.statSync(assetPath).isDirectory()) {
        return cdk.FileAssetPackaging.ZIP_DIRECTORY;
    }
    if (fs.statSync(assetPath).isFile()) {
        return cdk.FileAssetPackaging.FILE;
    }
    throw new Error(`Asset ${assetPath} is expected to be either a directory or a regular file`);
}
//# sourceMappingURL=data:application/json;base64,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