"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
/**
 * The record type.
 */
var RecordType;
(function (RecordType) {
    /**
     * route traffic to a resource, such as a web server, using an IPv4 address in dotted decimal
     * notation
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AFormat
     */
    RecordType["A"] = "A";
    /**
     * route traffic to a resource, such as a web server, using an IPv6 address in colon-separated
     * hexadecimal format
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#AAAAFormat
     */
    RecordType["AAAA"] = "AAAA";
    /**
     * A CAA record specifies which certificate authorities (CAs) are allowed to issue certificates
     * for a domain or subdomain
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CAAFormat
     */
    RecordType["CAA"] = "CAA";
    /**
     * A CNAME record maps DNS queries for the name of the current record, such as acme.example.com,
     * to another domain (example.com or example.net) or subdomain (acme.example.com or zenith.example.org).
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#CNAMEFormat
     */
    RecordType["CNAME"] = "CNAME";
    /**
     * An MX record specifies the names of your mail servers and, if you have two or more mail servers,
     * the priority order.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#MXFormat
     */
    RecordType["MX"] = "MX";
    /**
     * A Name Authority Pointer (NAPTR) is a type of record that is used by Dynamic Delegation Discovery
     * System (DDDS) applications to convert one value to another or to replace one value with another.
     * For example, one common use is to convert phone numbers into SIP URIs.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NAPTRFormat
     */
    RecordType["NAPTR"] = "NAPTR";
    /**
     * An NS record identifies the name servers for the hosted zone
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#NSFormat
     */
    RecordType["NS"] = "NS";
    /**
     * A PTR record maps an IP address to the corresponding domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#PTRFormat
     */
    RecordType["PTR"] = "PTR";
    /**
     * A start of authority (SOA) record provides information about a domain and the corresponding Amazon
     * Route 53 hosted zone
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SOAFormat
     */
    RecordType["SOA"] = "SOA";
    /**
     * SPF records were formerly used to verify the identity of the sender of email messages.
     * Instead of an SPF record, we recommend that you create a TXT record that contains the applicable value.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SPFFormat
     */
    RecordType["SPF"] = "SPF";
    /**
     * An SRV record Value element consists of four space-separated values. The first three values are
     * decimal numbers representing priority, weight, and port. The fourth value is a domain name.
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#SRVFormat
     */
    RecordType["SRV"] = "SRV";
    /**
     * A TXT record contains one or more strings that are enclosed in double quotation marks (").
     *
     * @see https://docs.aws.amazon.com/Route53/latest/DeveloperGuide/ResourceRecordTypes.html#TXTFormat
     */
    RecordType["TXT"] = "TXT";
})(RecordType = exports.RecordType || (exports.RecordType = {}));
/**
 * Type union for a record that accepts multiple types of target.
 */
class RecordTarget {
    /**
     *
     * @param values correspond with the chosen record type (e.g. for 'A' Type, specify one ore more IP addresses)
     * @param aliasTarget alias for targets such as CloudFront distribution to route traffic to
     */
    constructor(values, aliasTarget) {
        this.values = values;
        this.aliasTarget = aliasTarget;
    }
    /**
     * Use string values as target.
     */
    static fromValues(...values) {
        return new RecordTarget(values);
    }
    /**
     * Use an alias as target.
     */
    static fromAlias(aliasTarget) {
        return new RecordTarget(undefined, aliasTarget);
    }
    /**
     * Use ip addresses as target.
     */
    static fromIpAddresses(...ipAddresses) {
        return RecordTarget.fromValues(...ipAddresses);
    }
}
exports.RecordTarget = RecordTarget;
/**
 * A record set.
 */
class RecordSet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const ttl = props.target.aliasTarget ? undefined : ((props.ttl && props.ttl.toSeconds()) || 1800).toString();
        const recordSet = new route53_generated_1.CfnRecordSet(this, 'Resource', {
            hostedZoneId: props.zone.hostedZoneId,
            name: util_1.determineFullyQualifiedDomainName(props.recordName || props.zone.zoneName, props.zone),
            type: props.recordType,
            resourceRecords: props.target.values,
            aliasTarget: props.target.aliasTarget && props.target.aliasTarget.bind(this),
            ttl,
            comment: props.comment,
        });
        this.domainName = recordSet.ref;
    }
}
exports.RecordSet = RecordSet;
/**
 * Target for a DNS A Record
 *
 * @deprecated Use RecordTarget
 */
class AddressRecordTarget extends RecordTarget {
}
exports.AddressRecordTarget = AddressRecordTarget;
/**
 * A DNS A record
 *
 * @resource AWS::Route53::RecordSet
 */
class ARecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.A,
            target: props.target,
        });
    }
}
exports.ARecord = ARecord;
/**
 * A DNS AAAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class AaaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.AAAA,
            target: props.target,
        });
    }
}
exports.AaaaRecord = AaaaRecord;
/**
 * A DNS CNAME record
 *
 * @resource AWS::Route53::RecordSet
 */
class CnameRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CNAME,
            target: RecordTarget.fromValues(props.domainName),
        });
    }
}
exports.CnameRecord = CnameRecord;
/**
 * A DNS TXT record
 *
 * @resource AWS::Route53::RecordSet
 */
class TxtRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.TXT,
            target: RecordTarget.fromValues(...props.values.map(v => JSON.stringify(v))),
        });
    }
}
exports.TxtRecord = TxtRecord;
/**
 * A DNS SRV record
 *
 * @resource AWS::Route53::RecordSet
 */
class SrvRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.SRV,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.weight} ${v.port} ${v.hostName}`)),
        });
    }
}
exports.SrvRecord = SrvRecord;
/**
 * The CAA tag.
 */
var CaaTag;
(function (CaaTag) {
    /**
     * Explicity authorizes a single certificate authority to issue a
     * certificate (any type) for the hostname.
     */
    CaaTag["ISSUE"] = "issue";
    /**
     * Explicity authorizes a single certificate authority to issue a
     * wildcard certificate (and only wildcard) for the hostname.
     */
    CaaTag["ISSUEWILD"] = "issuewild";
    /**
     * Specifies a URL to which a certificate authority may report policy
     * violations.
     */
    CaaTag["IODEF"] = "iodef";
})(CaaTag = exports.CaaTag || (exports.CaaTag = {}));
/**
 * A DNS CAA record
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.CAA,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.flag} ${v.tag} "${v.value}"`)),
        });
    }
}
exports.CaaRecord = CaaRecord;
/**
 * A DNS Amazon CAA record.
 *
 * A CAA record to restrict certificate authorities allowed
 * to issue certificates for a domain to Amazon only.
 *
 * @resource AWS::Route53::RecordSet
 */
class CaaAmazonRecord extends CaaRecord {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            values: [
                {
                    flag: 0,
                    tag: CaaTag.ISSUE,
                    value: 'amazon.com',
                },
            ],
        });
    }
}
exports.CaaAmazonRecord = CaaAmazonRecord;
/**
 * A DNS MX record
 *
 * @resource AWS::Route53::RecordSet
 */
class MxRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.MX,
            target: RecordTarget.fromValues(...props.values.map(v => `${v.priority} ${v.hostName}`)),
        });
    }
}
exports.MxRecord = MxRecord;
/**
 * A record to delegate further lookups to a different set of name servers.
 */
class ZoneDelegationRecord extends RecordSet {
    constructor(scope, id, props) {
        super(scope, id, {
            ...props,
            recordType: RecordType.NS,
            target: RecordTarget.fromValues(...core_1.Token.isUnresolved(props.nameServers)
                ? props.nameServers // Can't map a string-array token!
                : props.nameServers.map(ns => (core_1.Token.isUnresolved(ns) || ns.endsWith('.')) ? ns : `${ns}.`)),
            ttl: props.ttl || core_1.Duration.days(2),
        });
    }
}
exports.ZoneDelegationRecord = ZoneDelegationRecord;
//# sourceMappingURL=data:application/json;base64,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