"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_certificatemanager_1 = require("../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const aws_cloudfront_1 = require("../../aws-cloudfront"); // Automatically re-written from '@aws-cdk/aws-cloudfront'
const aws_route53_1 = require("../../aws-route53"); // Automatically re-written from '@aws-cdk/aws-route53'
const aws_route53_targets_1 = require("../../aws-route53-targets"); // Automatically re-written from '@aws-cdk/aws-route53-targets'
const aws_s3_1 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const crypto = require("crypto");
/**
 * Allows creating a domainA -> domainB redirect using CloudFront and S3.
 * You can specify multiple domains to be redirected.
 */
class HttpsRedirect extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const domainNames = props.recordNames || [props.zone.zoneName];
        const redirectCertArn = props.certificate ? props.certificate.certificateArn : new aws_certificatemanager_1.DnsValidatedCertificate(this, 'RedirectCertificate', {
            domainName: domainNames[0],
            subjectAlternativeNames: domainNames,
            hostedZone: props.zone,
            region: 'us-east-1',
        }).certificateArn;
        const redirectBucket = new aws_s3_1.Bucket(this, 'RedirectBucket', {
            websiteRedirect: {
                hostName: props.targetDomain,
                protocol: aws_s3_1.RedirectProtocol.HTTPS,
            },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const redirectDist = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'RedirectDistribution', {
            defaultRootObject: '',
            originConfigs: [{
                    behaviors: [{ isDefaultBehavior: true }],
                    customOriginSource: {
                        domainName: redirectBucket.bucketWebsiteDomainName,
                        originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY,
                    },
                }],
            aliasConfiguration: {
                acmCertRef: redirectCertArn,
                names: domainNames,
            },
            comment: `Redirect to ${props.targetDomain} from ${domainNames.join(', ')}`,
            priceClass: aws_cloudfront_1.PriceClass.PRICE_CLASS_ALL,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        });
        domainNames.forEach((domainName) => {
            const hash = crypto.createHash('md5').update(domainName).digest('hex').substr(0, 6);
            new aws_route53_1.ARecord(this, `RedirectAliasRecord${hash}`, {
                recordName: domainName,
                zone: props.zone,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(redirectDist)),
            });
        });
    }
}
exports.HttpsRedirect = HttpsRedirect;
//# sourceMappingURL=data:application/json;base64,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