"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const props_1 = require("./props");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            sep: ':',
            resourceName: this.instanceIdentifier,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
exports.DatabaseInstanceBase = DatabaseInstanceBase;
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
class DatabaseInstanceEngine extends props_1.DatabaseClusterEngine {
    constructor() {
        super(...arguments);
        /* tslint:enable max-line-length */
        /** To make it a compile-time error to pass a DatabaseClusterEngine where a DatabaseInstanceEngine is expected. */
        this.isDatabaseInstanceEngine = true;
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/* tslint:disable max-line-length */
DatabaseInstanceEngine.MARIADB = new DatabaseInstanceEngine('mariadb', secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER, [
    { engineMajorVersion: '10.0', parameterGroupFamily: 'mariadb10.0' },
    { engineMajorVersion: '10.1', parameterGroupFamily: 'mariadb10.1' },
    { engineMajorVersion: '10.2', parameterGroupFamily: 'mariadb10.2' },
    { engineMajorVersion: '10.3', parameterGroupFamily: 'mariadb10.3' },
]);
DatabaseInstanceEngine.MYSQL = new DatabaseInstanceEngine('mysql', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.6', parameterGroupFamily: 'mysql5.6' },
    { engineMajorVersion: '5.7', parameterGroupFamily: 'mysql5.7' },
    { engineMajorVersion: '8.0', parameterGroupFamily: 'mysql8.0' },
]);
DatabaseInstanceEngine.ORACLE_EE = new DatabaseInstanceEngine('oracle-ee', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-ee-11.2' },
    { engineMajorVersion: '12.1', parameterGroupFamily: 'oracle-ee-12.1' },
    { engineMajorVersion: '12.2', parameterGroupFamily: 'oracle-ee-12.2' },
    { engineMajorVersion: '18', parameterGroupFamily: 'oracle-ee-18' },
    { engineMajorVersion: '19', parameterGroupFamily: 'oracle-ee-19' },
]);
DatabaseInstanceEngine.ORACLE_SE2 = new DatabaseInstanceEngine('oracle-se2', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '12.1', parameterGroupFamily: 'oracle-se2-12.1' },
    { engineMajorVersion: '12.2', parameterGroupFamily: 'oracle-se2-12.2' },
    { engineMajorVersion: '18', parameterGroupFamily: 'oracle-se2-18' },
    { engineMajorVersion: '19', parameterGroupFamily: 'oracle-se2-19' },
]);
DatabaseInstanceEngine.ORACLE_SE1 = new DatabaseInstanceEngine('oracle-se1', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-se1-11.2' },
]);
DatabaseInstanceEngine.ORACLE_SE = new DatabaseInstanceEngine('oracle-se', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-se-11.2' },
]);
DatabaseInstanceEngine.POSTGRES = new DatabaseInstanceEngine('postgres', secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER, [
    { engineMajorVersion: '9.3', parameterGroupFamily: 'postgres9.3' },
    { engineMajorVersion: '9.4', parameterGroupFamily: 'postgres9.4' },
    { engineMajorVersion: '9.5', parameterGroupFamily: 'postgres9.5' },
    { engineMajorVersion: '9.6', parameterGroupFamily: 'postgres9.6' },
    { engineMajorVersion: '10', parameterGroupFamily: 'postgres10' },
    { engineMajorVersion: '11', parameterGroupFamily: 'postgres11' },
]);
DatabaseInstanceEngine.SQL_SERVER_EE = new DatabaseInstanceEngine('sqlserver-ee', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-ee-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-ee-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-ee-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-ee-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_SE = new DatabaseInstanceEngine('sqlserver-se', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-se-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-se-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-se-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-se-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_EX = new DatabaseInstanceEngine('sqlserver-ex', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-ex-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-ex-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-ex-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-ex-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_WEB = new DatabaseInstanceEngine('sqlserver-web', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-web-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-web-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-web-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-web-14.0' },
]);
/**
 * The license model.
 */
var LicenseModel;
(function (LicenseModel) {
    /**
     * License included.
     */
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    /**
     * Bring your own licencse.
     */
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    /**
     * General public license.
     */
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * The type of storage.
 */
var StorageType;
(function (StorageType) {
    /**
     * Standard.
     */
    StorageType["STANDARD"] = "standard";
    /**
     * General purpose (SSD).
     */
    StorageType["GP2"] = "gp2";
    /**
     * Provisioned IOPS (SSD).
     */
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * The retention period for Performance Insight.
 */
var PerformanceInsightRetention;
(function (PerformanceInsightRetention) {
    /**
     * Default retention period of 7 days.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["DEFAULT"] = 7] = "DEFAULT";
    /**
     * Long term retention period of 2 years.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["LONG_TERM"] = 731] = "LONG_TERM";
})(PerformanceInsightRetention = exports.PerformanceInsightRetention || (exports.PerformanceInsightRetention = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcPlacement = props.vpcPlacement;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcPlacement);
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'SubnetGroup', {
            dbSubnetGroupDescription: `Subnet group for ${this.node.id} database`,
            subnetIds,
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.numberValue({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const deletionProtection = props.deletionProtection !== undefined ? props.deletionProtection : true;
        const storageType = props.storageType || StorageType.GP2;
        const iops = storageType === StorageType.IO1 ? (props.iops || 1000) : undefined;
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: props.backupRetention ? props.backupRetention.toDays() : undefined,
            copyTagsToSnapshot: props.copyTagsToSnapshot !== undefined ? props.copyTagsToSnapshot : true,
            dbInstanceClass: `db.${props.instanceClass}`,
            dbInstanceIdentifier: props.instanceIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection,
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: props.iamAuthentication,
            enablePerformanceInsights: props.enablePerformanceInsights,
            iops,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            multiAz: props.multiAz,
            optionGroupName: props.optionGroup && props.optionGroup.optionGroupName,
            performanceInsightsKmsKeyId: props.enablePerformanceInsights
                ? props.performanceInsightKmsKey && props.performanceInsightKmsKey.keyArn
                : undefined,
            performanceInsightsRetentionPeriod: props.enablePerformanceInsights
                ? (props.performanceInsightRetention || PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: props.port ? props.port.toString() : undefined,
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: props.vpcPlacement && props.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC,
            storageType,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
        };
    }
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new lambda.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const timezoneSupport = [DatabaseInstanceEngine.SQL_SERVER_EE, DatabaseInstanceEngine.SQL_SERVER_EX,
            DatabaseInstanceEngine.SQL_SERVER_SE, DatabaseInstanceEngine.SQL_SERVER_WEB];
        if (props.timezone && !timezoneSupport.includes(props.engine)) {
            throw new Error(`timezone property can be configured only for Microsoft SQL Server, not ${props.engine.name}`);
        }
        this.sourceCfnProps = {
            ...this.newCfnProps,
            allocatedStorage: props.allocatedStorage ? props.allocatedStorage.toString() : '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            dbParameterGroupName: props.parameterGroup && props.parameterGroup.parameterGroupName,
            engine: props.engine.name,
            engineVersion: props.engineVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
        };
    }
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this instance.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (!props.masterUserPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.secretKmsKey,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.kmsKey && props.kmsKey.keyArn,
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUsername,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
            storageEncrypted: props.kmsKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstance = DatabaseInstance;
/**
 * A database instance restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (props.generateMasterUserPassword) {
            if (!props.masterUsername) { // We need the master username to include it in the generated secret
                throw new Error('`masterUsername` must be specified when `generateMasterUserPassword` is set to true.');
            }
            if (props.masterUserPassword) {
                throw new Error('Cannot specify `masterUserPassword` when `generateMasterUserPassword` is set to true.');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.secretKmsKey,
            });
        }
        else {
            if (props.masterUsername) { // It's not possible to change the master username of a RDS instance
                throw new Error('Cannot specify `masterUsername` when `generateMasterUserPassword` is set to false.');
            }
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
/**
 * A read replica database instance.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: props.kmsKey && props.kmsKey.keyArn,
            storageEncrypted: props.kmsKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        this.setLogRetention();
    }
}
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
function applyInstanceDeletionPolicy(cfnDbInstance, removalPolicy) {
    if (!removalPolicy) {
        // the default DeletionPolicy is 'Snapshot', which is fine,
        // but we should also make it 'Snapshot' for UpdateReplace policy
        cfnDbInstance.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
    }
    else {
        // just apply whatever removal policy the customer explicitly provided
        cfnDbInstance.applyRemovalPolicy(removalPolicy);
    }
}
//# sourceMappingURL=data:application/json;base64,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