"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const parameter_group_1 = require("./parameter-group");
const props_1 = require("./props");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_CLUSTER,
        };
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Identifiers of the replicas
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual replica.
         */
        this.instanceEndpoints = [];
        this.vpc = props.instanceProps.vpc;
        this.vpcSubnets = props.instanceProps.vpcSubnets;
        const { subnetIds } = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets);
        // Cannot test whether the subnets are in different AZs, but at least we can test the amount.
        if (subnetIds.length < 2) {
            this.node.addError(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'Subnets', {
            dbSubnetGroupDescription: `Subnets for ${id} database`,
            subnetIds,
        });
        if (props.removalPolicy === core_1.RemovalPolicy.RETAIN) {
            subnetGroup.applyRemovalPolicy(core_1.RemovalPolicy.RETAIN);
        }
        const securityGroup = props.instanceProps.securityGroup !== undefined ?
            props.instanceProps.securityGroup : new ec2.SecurityGroup(this, 'SecurityGroup', {
            description: 'RDS security group',
            vpc: props.instanceProps.vpc,
        });
        this.securityGroupId = securityGroup.securityGroupId;
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.kmsKey,
            });
        }
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        let s3ImportRole = props.s3ImportRole;
        if (props.s3ImportBuckets && props.s3ImportBuckets.length > 0) {
            if (props.s3ImportRole) {
                throw new Error('Only one of s3ImportRole or s3ImportBuckets must be specified, not both.');
            }
            s3ImportRole = new aws_iam_1.Role(this, 'S3ImportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
            });
            for (const bucket of props.s3ImportBuckets) {
                bucket.grantRead(s3ImportRole);
            }
        }
        let s3ExportRole = props.s3ExportRole;
        if (props.s3ExportBuckets && props.s3ExportBuckets.length > 0) {
            if (props.s3ExportRole) {
                throw new Error('Only one of s3ExportRole or s3ExportBuckets must be specified, not both.');
            }
            s3ExportRole = new aws_iam_1.Role(this, 'S3ExportRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('rds.amazonaws.com'),
            });
            for (const bucket of props.s3ExportBuckets) {
                bucket.grantReadWrite(s3ExportRole);
            }
        }
        let clusterParameterGroup = props.parameterGroup;
        const clusterAssociatedRoles = [];
        if (s3ImportRole || s3ExportRole) {
            if (s3ImportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ImportRole.roleArn });
            }
            if (s3ExportRole) {
                clusterAssociatedRoles.push({ roleArn: s3ExportRole.roleArn });
            }
            // MySQL requires the associated roles to be specified as cluster parameters as well, PostgreSQL does not
            if (props.engine === props_1.DatabaseClusterEngine.AURORA || props.engine === props_1.DatabaseClusterEngine.AURORA_MYSQL) {
                if (!clusterParameterGroup) {
                    const parameterGroupFamily = props.engine.parameterGroupFamily(props.engineVersion);
                    if (!parameterGroupFamily) {
                        throw new Error(`No parameter group family found for database engine ${props.engine.name} with version ${props.engineVersion}.` +
                            'Failed to set the correct cluster parameters for s3 import and export roles.');
                    }
                    clusterParameterGroup = new parameter_group_1.ClusterParameterGroup(this, 'ClusterParameterGroup', {
                        family: parameterGroupFamily,
                    });
                }
                if (clusterParameterGroup instanceof parameter_group_1.ClusterParameterGroup) { // ignore imported ClusterParameterGroup
                    if (s3ImportRole) {
                        clusterParameterGroup.addParameter('aurora_load_from_s3_role', s3ImportRole.roleArn);
                    }
                    if (s3ExportRole) {
                        clusterParameterGroup.addParameter('aurora_select_into_s3_role', s3ExportRole.roleArn);
                    }
                }
            }
        }
        const cluster = new rds_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engine: props.engine.name,
            engineVersion: props.engineVersion,
            dbClusterIdentifier: props.clusterIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            vpcSecurityGroupIds: [this.securityGroupId],
            port: props.port,
            dbClusterParameterGroupName: clusterParameterGroup && clusterParameterGroup.parameterGroupName,
            associatedRoles: clusterAssociatedRoles.length > 0 ? clusterAssociatedRoles : undefined,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : (props.masterUser.password
                    ? props.masterUser.password.toString()
                    : undefined),
            backupRetentionPeriod: props.backup && props.backup.retention && props.backup.retention.toDays(),
            preferredBackupWindow: props.backup && props.backup.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            databaseName: props.defaultDatabaseName,
            // Encryption
            kmsKeyId: props.kmsKey && props.kmsKey.keyArn,
            storageEncrypted: props.kmsKey ? true : props.storageEncrypted,
        });
        // if removalPolicy was not specified,
        // leave it as the default, which is Snapshot
        if (props.removalPolicy) {
            cluster.applyRemovalPolicy(props.removalPolicy);
        }
        else {
            // The CFN default makes sense for DeletionPolicy,
            // but doesn't cover UpdateReplacePolicy.
            // Fix that here.
            cluster.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
        }
        this.clusterIdentifier = cluster.ref;
        // create a number token that represents the port of the cluster
        const portAttribute = core_1.Token.asNumber(cluster.attrEndpointPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(cluster.attrEndpointAddress, portAttribute);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(cluster.attrReadEndpointAddress, portAttribute);
        if (secret) {
            this.secret = secret.attach(this);
        }
        const instanceCount = props.instances != null ? props.instances : 2;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        // Get the actual subnet objects so we can depend on internet connectivity.
        const internetConnected = props.instanceProps.vpc.selectSubnets(props.instanceProps.vpcSubnets).internetConnectivityEstablished;
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new aws_iam_1.Role(this, 'MonitoringRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole'),
                ],
            });
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}` :
                props.clusterIdentifier != null ? `${props.clusterIdentifier}instance${instanceIndex}` :
                    undefined;
            const publiclyAccessible = props.instanceProps.vpcSubnets && props.instanceProps.vpcSubnets.subnetType === ec2.SubnetType.PUBLIC;
            const instance = new rds_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                engine: props.engine.name,
                engineVersion: props.engineVersion,
                dbClusterIdentifier: cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: databaseInstanceType(props.instanceProps.instanceType),
                publiclyAccessible,
                // This is already set on the Cluster. Unclear to me whether it should be repeated or not. Better yes.
                dbSubnetGroupName: subnetGroup.ref,
                dbParameterGroupName: props.instanceProps.parameterGroup && props.instanceProps.parameterGroup.parameterGroupName,
                monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
                monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            });
            // If removalPolicy isn't explicitly set,
            // it's Snapshot for Cluster.
            // Because of that, in this case,
            // we can safely use the CFN default of Delete for DbInstances with dbClusterIdentifier set.
            if (props.removalPolicy) {
                instance.applyRemovalPolicy(props.removalPolicy);
            }
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnected);
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute));
        }
        const defaultPort = ec2.Port.tcp(this.clusterEndpoint.port);
        this.connections = new ec2.Connections({ securityGroups: [securityGroup], defaultPort });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.instanceIdentifiers = [];
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.instanceEndpoints = attrs.instanceEndpointAddresses.map(a => new endpoint_1.Endpoint(a, attrs.port));
                this.securityGroupId = attrs.securityGroup.securityGroupId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
}
exports.DatabaseCluster = DatabaseCluster;
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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