"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled
     * as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up) {
        this.lambdaFunction.node.addDependency(...up);
    }
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using
     * node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down) {
        down.node.addDependency(this.lambdaFunction);
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * @internal
     */
    _functionNode() {
        return this.lambdaFunction.node;
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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