"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
class Code {
    /**
     * @returns `LambdaS3Code` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
    /**
     * @deprecated use `fromBucket`
     */
    static bucket(bucket, key, objectVersion) {
        return this.fromBucket(bucket, key, objectVersion);
    }
    /**
     * @returns `LambdaInlineCode` with inline code.
     * @param code The actual handler code (limited to 4KiB)
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * @deprecated use `fromInline`
     */
    static inline(code) {
        return this.fromInline(code);
    }
    /**
     * Loads the function code from a local disk path.
     *
     * @param path Either a directory with the Lambda code bundle or a .zip file
     */
    static fromAsset(path, options) {
        return new AssetCode(path, options);
    }
    /**
     * @deprecated use `fromAsset`
     */
    static asset(path) {
        return this.fromAsset(path);
    }
    /**
     * Creates a new Lambda source defined using CloudFormation parameters.
     *
     * @returns a new instance of `CfnParametersCode`
     * @param props optional construction properties of {@link CfnParametersCode}
     */
    static fromCfnParameters(props) {
        return new CfnParametersCode(props);
    }
    /**
     * @deprecated use `fromCfnParameters`
     */
    static cfnParameters(props) {
        return this.fromCfnParameters(props);
    }
    /**
     * Called after the CFN function resource has been created to allow the code
     * class to bind to it. Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindToResource(_resource, _options) {
        return;
    }
}
exports.Code = Code;
/**
 * Lambda code from an S3 archive.
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        this.isInline = false;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
/**
 * Lambda code from an inline string (limited to 4KiB).
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        this.isInline = true;
        if (code.length === 0) {
            throw new Error('Lambda inline code cannot be empty');
        }
        if (code.length > 4096) {
            throw new Error('Lambda source is too large, must be <= 4096 but is ' + code.length);
        }
    }
    bind(_scope) {
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
/**
 * Lambda code from a local directory.
 */
class AssetCode extends Code {
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        this.isInline = false;
    }
    bind(scope) {
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.path,
                ...this.options,
            });
        }
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    bindToResource(resource, options = {}) {
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const resourceProperty = options.resourceProperty || 'Code';
        // https://github.com/aws/aws-cdk/issues/1432
        this.asset.addResourceMetadata(resource, resourceProperty);
    }
}
exports.AssetCode = AssetCode;
/**
 * Lambda code defined using 2 CloudFormation parameters.
 * Useful when you don't have access to the code of your Lambda from your CDK code, so you can't use Assets,
 * and you want to deploy the Lambda in a CodePipeline, using CloudFormation Actions -
 * you can fill the parameters using the {@link #assign} method.
 */
class CfnParametersCode extends Code {
    constructor(props = {}) {
        super();
        this.isInline = false;
        this._bucketNameParam = props.bucketNameParam;
        this._objectKeyParam = props.objectKeyParam;
    }
    bind(scope) {
        if (!this._bucketNameParam) {
            this._bucketNameParam = new cdk.CfnParameter(scope, 'LambdaSourceBucketNameParameter', {
                type: 'String',
            });
        }
        if (!this._objectKeyParam) {
            this._objectKeyParam = new cdk.CfnParameter(scope, 'LambdaSourceObjectKeyParameter', {
                type: 'String',
            });
        }
        return {
            s3Location: {
                bucketName: this._bucketNameParam.valueAsString,
                objectKey: this._objectKeyParam.valueAsString,
            },
        };
    }
    /**
     * Create a parameters map from this instance's CloudFormation parameters.
     *
     * It returns a map with 2 keys that correspond to the names of the parameters defined in this Lambda code,
     * and as values it contains the appropriate expressions pointing at the provided S3 location
     * (most likely, obtained from a CodePipeline Artifact by calling the `artifact.s3Location` method).
     * The result should be provided to the CloudFormation Action
     * that is deploying the Stack that the Lambda with this code is part of,
     * in the `parameterOverrides` property.
     *
     * @param location the location of the object in S3 that represents the Lambda code
     */
    assign(location) {
        const ret = {};
        ret[this.bucketNameParam] = location.bucketName;
        ret[this.objectKeyParam] = location.objectKey;
        return ret;
    }
    get bucketNameParam() {
        if (this._bucketNameParam) {
            return this._bucketNameParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the bucketNameParam property');
        }
    }
    get objectKeyParam() {
        if (this._objectKeyParam) {
            return this._objectKeyParam.logicalId;
        }
        else {
            throw new Error('Pass CfnParametersCode to a Lambda Function before accessing the objectKeyParam property');
        }
    }
}
exports.CfnParametersCode = CfnParametersCode;
//# sourceMappingURL=data:application/json;base64,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