"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const builder_1 = require("./builder");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        var _a;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        const entry = findEntry(id, props.entry);
        const handler = props.handler || 'handler';
        const buildDir = props.buildDir || path.join(path.dirname(entry), '.build');
        const handlerDir = path.join(buildDir, crypto.createHash('sha256').update(entry).digest('hex'));
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = props.runtime || defaultRunTime;
        const projectRoot = (_a = props.projectRoot) !== null && _a !== void 0 ? _a : util_1.findGitPath();
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        // Build with Parcel
        const builder = new builder_1.Builder({
            entry,
            outDir: handlerDir,
            global: handler,
            minify: props.minify,
            sourceMaps: props.sourceMaps,
            cacheDir: props.cacheDir,
            nodeVersion: extractVersion(runtime),
            nodeDockerTag: props.nodeDockerTag || `${process.versions.node}-alpine`,
            projectRoot: path.resolve(projectRoot),
            environment: props.containerEnvironment,
        });
        builder.build();
        super(scope, id, {
            ...props,
            runtime,
            code: lambda.Code.fromAsset(handlerDir),
            handler: `index.${handler}`,
        });
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(js|ts)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
/**
 * Extracts the version from the runtime
 */
function extractVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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