"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const targets = require("../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const event_bridge_1 = require("./event-bridge");
/**
 * Use a Lambda function as a Lambda destination
 */
class LambdaDestination {
    constructor(fn, options = {}) {
        this.fn = fn;
        this.options = options;
    }
    /**
     * Returns a destination configuration
     */
    bind(scope, fn, options) {
        // Normal Lambda destination (full invocation record)
        if (!this.options.responseOnly) {
            // deduplicated automatically
            this.fn.grantInvoke(fn);
            return {
                destination: this.fn.functionArn,
            };
        }
        // Otherwise add rule to extract the response payload and use EventBridge
        // as destination
        if (!options) { // `options` added to bind() as optionnal to avoid breaking change
            throw new Error('Options must be defined when using `responseOnly`.');
        }
        // Match invocation result of the source function (`fn`) and use it
        // to trigger the destination function (`this.fn`).
        new events.Rule(scope, options.type, {
            eventPattern: {
                detailType: [`Lambda Function Invocation Result - ${options.type}`],
                resources: [`${fn.functionArn}:$LATEST`],
                source: ['lambda'],
            },
            targets: [
                new targets.LambdaFunction(this.fn, {
                    event: events.RuleTargetInput.fromEventPath('$.detail.responsePayload'),
                }),
            ],
        });
        const destination = new event_bridge_1.EventBridgeDestination(); // Use default event bus here
        return destination.bind(scope, fn);
    }
}
exports.LambdaDestination = LambdaDestination;
//# sourceMappingURL=data:application/json;base64,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