"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const kinesis_generated_1 = require("./kinesis.generated");
const READ_OPERATIONS = [
    'kinesis:DescribeStreamSummary',
    'kinesis:GetRecords',
    'kinesis:GetShardIterator',
    'kinesis:ListShards',
    'kinesis:SubscribeToShard',
];
const WRITE_OPERATIONS = [
    'kinesis:ListShards',
    'kinesis:PutRecord',
    'kinesis:PutRecords',
];
/**
 * Represents a Kinesis Stream.
 */
class StreamBase extends core_1.Resource {
    /**
     * Grant write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, ...READ_OPERATIONS);
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant read permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to ues the key to decrypt the
     * contents of the stream will also be granted.
     */
    grantWrite(grantee) {
        var _a;
        const ret = this.grant(grantee, ...WRITE_OPERATIONS);
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncrypt(grantee);
        return ret;
    }
    /**
     * Grants read/write permissions for this stream and its contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     */
    grantReadWrite(grantee) {
        var _a;
        const ret = this.grant(grantee, ...Array.from(new Set([...READ_OPERATIONS, ...WRITE_OPERATIONS])));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(grantee);
        return ret;
    }
    /**
     * Grant the indicated permissions on this stream to the given IAM principal (Role/Group/User).
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.streamArn],
            scope: this,
        });
    }
}
/**
 * A Kinesis stream. Can be encrypted with a KMS key.
 */
class Stream extends StreamBase {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.streamName,
        });
        const shardCount = props.shardCount || 1;
        const retentionPeriodHours = (_b = (_a = props.retentionPeriod) === null || _a === void 0 ? void 0 : _a.toHours()) !== null && _b !== void 0 ? _b : 24;
        if (retentionPeriodHours < 24 || retentionPeriodHours > 168) {
            throw new Error(`retentionPeriod must be between 24 and 168 hours. Received ${retentionPeriodHours}`);
        }
        const { streamEncryption, encryptionKey } = this.parseEncryption(props);
        this.stream = new kinesis_generated_1.CfnStream(this, 'Resource', {
            name: this.physicalName,
            retentionPeriodHours,
            shardCount,
            streamEncryption,
        });
        this.streamArn = this.getResourceArnAttribute(this.stream.attrArn, {
            service: 'kinesis',
            resource: 'stream',
            resourceName: this.physicalName,
        });
        this.streamName = this.getResourceNameAttribute(this.stream.ref);
        this.encryptionKey = encryptionKey;
    }
    /**
     * Import an existing Kinesis Stream provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name
     * @param streamArn Stream ARN (i.e. arn:aws:kinesis:<region>:<account-id>:stream/Foo)
     */
    static fromStreamArn(scope, id, streamArn) {
        return Stream.fromStreamAttributes(scope, id, { streamArn });
    }
    /**
     * Creates a Stream construct that represents an external stream.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs Stream import properties
     */
    static fromStreamAttributes(scope, id, attrs) {
        class Import extends StreamBase {
            constructor() {
                super(...arguments);
                this.streamArn = attrs.streamArn;
                this.streamName = core_1.Stack.of(scope).parseArn(attrs.streamArn).resourceName;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Set up key properties and return the Stream encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _a;
        // if encryption properties are not set, default to KMS in regions where KMS is available
        if (!props.encryption && !props.encryptionKey) {
            const conditionName = 'AwsCdkKinesisEncryptedStreamsUnsupportedRegions';
            const existing = core_1.Stack.of(this).node.tryFindChild(conditionName);
            // create a single condition for the Stack
            if (!existing) {
                new core_1.CfnCondition(core_1.Stack.of(this), conditionName, {
                    expression: core_1.Fn.conditionOr(core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-north-1'), core_1.Fn.conditionEquals(core_1.Aws.REGION, 'cn-northwest-1')),
                });
            }
            return {
                streamEncryption: core_1.Fn.conditionIf(conditionName, core_1.Aws.NO_VALUE, { EncryptionType: 'KMS', KeyId: 'alias/aws/kinesis' }),
            };
        }
        // default based on whether encryption key is specified
        const encryptionType = (_a = props.encryption) !== null && _a !== void 0 ? _a : (props.encryptionKey ? StreamEncryption.KMS : StreamEncryption.UNENCRYPTED);
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== StreamEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === StreamEncryption.UNENCRYPTED) {
            return {};
        }
        if (encryptionType === StreamEncryption.MANAGED) {
            const encryption = { encryptionType: 'KMS', keyId: 'alias/aws/kinesis' };
            return { streamEncryption: encryption };
        }
        if (encryptionType === StreamEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const streamEncryption = {
                encryptionType: 'KMS',
                keyId: encryptionKey.keyArn,
            };
            return { encryptionKey, streamEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
}
exports.Stream = Stream;
/**
 * What kind of server-side encryption to apply to this stream
 */
var StreamEncryption;
(function (StreamEncryption) {
    /**
     * Records in the stream are not encrypted.
     */
    StreamEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    StreamEncryption["KMS"] = "KMS";
    /**
     * Server-side encryption with a master key managed by Amazon Kinesis
     */
    StreamEncryption["MANAGED"] = "MANAGED";
})(StreamEncryption = exports.StreamEncryption || (exports.StreamEncryption = {}));
//# sourceMappingURL=data:application/json;base64,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