"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const path = require("path");
const RESOURCE_TYPE = 'Custom::AWSCDKOpenIdConnectProvider';
/**
 * IAM OIDC identity providers are entities in IAM that describe an external
 * identity provider (IdP) service that supports the OpenID Connect (OIDC)
 * standard, such as Google or Salesforce. You use an IAM OIDC identity provider
 * when you want to establish trust between an OIDC-compatible IdP and your AWS
 * account. This is useful when creating a mobile app or web application that
 * requires access to AWS resources, but you don't want to create custom sign-in
 * code or manage your own user identities.
 *
 * @see http://openid.net/connect
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc.html
 *
 * @experimental
 */
class OpenIdConnectProvider extends core_1.Construct {
    /**
     * Defines an OpenID Connect provider.
     * @param scope The definition scope
     * @param id Construct ID
     * @param props Initialization properties
     */
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: RESOURCE_TYPE,
            serviceToken: this.getOrCreateProvider(),
            properties: {
                ClientIDList: props.clientIds,
                ThumbprintList: props.thumbprints,
                Url: props.url,
            },
        });
        this.openIdConnectProviderArn = core_1.Token.asString(resource.ref);
    }
    /**
     * Imports an Open ID connect provider from an ARN.
     * @param scope The definition scope
     * @param id ID of the construct
     * @param openIdConnectProviderArn the ARN to import
     */
    static fromOpenIdConnectProviderArn(scope, id, openIdConnectProviderArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.openIdConnectProviderArn = openIdConnectProviderArn;
            }
        }
        return new Import(scope, id);
    }
    get stack() { return core_1.Stack.of(this); }
    getOrCreateProvider() {
        return core_1.CustomResourceProvider.getOrCreate(this, RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'oidc-provider'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                {
                    Effect: 'Allow',
                    Resource: '*',
                    Action: [
                        'iam:CreateOpenIDConnectProvider',
                        'iam:DeleteOpenIDConnectProvider',
                        'iam:UpdateOpenIDConnectProviderThumbprint',
                        'iam:AddClientIDToOpenIDConnectProvider',
                        'iam:RemoveClientIDFromOpenIDConnectProvider',
                    ],
                },
            ],
        });
    }
}
exports.OpenIdConnectProvider = OpenIdConnectProvider;
//# sourceMappingURL=data:application/json;base64,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