"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const glue_generated_1 = require("./glue.generated");
/**
 * A Glue database.
 */
class Database extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.databaseName,
        });
        let databaseInput = {
            name: props.databaseName,
        };
        if (props.locationUri !== undefined) {
            validateLocationUri(props.locationUri);
            this.locationUri = props.locationUri;
            databaseInput = {
                locationUri: this.locationUri,
                ...databaseInput,
            };
        }
        this.catalogId = core_1.Stack.of(this).account;
        const resource = new glue_generated_1.CfnDatabase(this, 'Resource', {
            catalogId: this.catalogId,
            databaseInput,
        });
        // see https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html#data-catalog-resource-arns
        this.databaseName = this.getResourceNameAttribute(resource.ref);
        this.databaseArn = this.stack.formatArn({
            service: 'glue',
            resource: 'database',
            resourceName: this.databaseName,
        });
        // catalogId is implicitly the accountId, which is why we don't pass the catalogId here
        this.catalogArn = core_1.Stack.of(this).formatArn({
            service: 'glue',
            resource: 'catalog',
        });
    }
    static fromDatabaseArn(scope, id, databaseArn) {
        const stack = core_1.Stack.of(scope);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.databaseArn = databaseArn;
                this.databaseName = stack.parseArn(databaseArn).resourceName;
                this.catalogArn = stack.formatArn({ service: 'glue', resource: 'catalog' });
                this.catalogId = stack.account;
            }
        }
        return new Import(scope, id);
    }
}
exports.Database = Database;
function validateLocationUri(locationUri) {
    if (locationUri.length < 1 || locationUri.length > 1024) {
        throw new Error(`locationUri length must be (inclusively) between 1 and 1024, but was ${locationUri.length}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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