"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * The input to send to the event target
 */
class RuleTargetInput {
    /**
     * Pass text to the event target
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromText(text) {
        return new FieldAwareEventInput(text, InputType.Text);
    }
    /**
     * Pass text to the event target, splitting on newlines.
     *
     * This is only useful when passing to a target that does not
     * take a single argument.
     *
     * May contain strings returned by EventField.from() to substitute in parts
     * of the matched event.
     */
    static fromMultilineText(text) {
        return new FieldAwareEventInput(text, InputType.Multiline);
    }
    /**
     * Pass a JSON object to the event target
     *
     * May contain strings returned by EventField.from() to substitute in parts of the
     * matched event.
     */
    static fromObject(obj) {
        return new FieldAwareEventInput(obj, InputType.Object);
    }
    /**
     * Take the event target input from a path in the event JSON
     */
    static fromEventPath(path) {
        return new LiteralEventInput({ inputPath: path });
    }
    constructor() {
    }
}
exports.RuleTargetInput = RuleTargetInput;
/**
 * Event Input that is directly derived from the construct
 */
class LiteralEventInput extends RuleTargetInput {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Return the input properties for this input object
     */
    bind(_rule) {
        return this.props;
    }
}
/**
 * Input object that can contain field replacements
 *
 * Evaluation is done in the bind() method because token resolution
 * requires access to the construct tree.
 *
 * Multiple tokens that use the same path will use the same substitution
 * key.
 *
 * One weird exception: if we're in object context, we MUST skip the quotes
 * around the placeholder. I assume this is so once a trivial string replace is
 * done later on by CWE, numbers are still numbers.
 *
 * So in string context:
 *
 *    "this is a string with a <field>"
 *
 * But in object context:
 *
 *    "{ \"this is the\": <field> }"
 *
 * To achieve the latter, we postprocess the JSON string to remove the surrounding
 * quotes by using a string replace.
 */
class FieldAwareEventInput extends RuleTargetInput {
    constructor(input, inputType) {
        super();
        this.input = input;
        this.inputType = inputType;
    }
    bind(rule) {
        let fieldCounter = 0;
        const pathToKey = new Map();
        const inputPathsMap = {};
        function keyForField(f) {
            const existing = pathToKey.get(f.path);
            if (existing !== undefined) {
                return existing;
            }
            fieldCounter += 1;
            const key = f.displayHint || `f${fieldCounter}`;
            pathToKey.set(f.path, key);
            return key;
        }
        const self = this;
        class EventFieldReplacer extends core_1.DefaultTokenResolver {
            constructor() {
                super(new core_1.StringConcat());
            }
            resolveToken(t, _context) {
                if (!isEventField(t)) {
                    return core_1.Token.asString(t);
                }
                const key = keyForField(t);
                if (inputPathsMap[key] && inputPathsMap[key] !== t.path) {
                    throw new Error(`Single key '${key}' is used for two different JSON paths: '${t.path}' and '${inputPathsMap[key]}'`);
                }
                inputPathsMap[key] = t.path;
                return self.keyPlaceholder(key);
            }
        }
        const stack = core_1.Stack.of(rule);
        let resolved;
        if (this.inputType === InputType.Multiline) {
            // JSONify individual lines
            resolved = core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            });
            resolved = resolved.split('\n').map(stack.toJsonString).join('\n');
        }
        else {
            resolved = stack.toJsonString(core_1.Tokenization.resolve(this.input, {
                scope: rule,
                resolver: new EventFieldReplacer(),
            }));
        }
        if (Object.keys(inputPathsMap).length === 0) {
            // Nothing special, just return 'input'
            return { input: resolved };
        }
        return {
            inputTemplate: this.unquoteKeyPlaceholders(resolved),
            inputPathsMap,
        };
    }
    /**
     * Return a template placeholder for the given key
     *
     * In object scope we'll need to get rid of surrounding quotes later on, so
     * return a bracing that's unlikely to occur naturally (like tokens).
     */
    keyPlaceholder(key) {
        if (this.inputType !== InputType.Object) {
            return `<${key}>`;
        }
        return UNLIKELY_OPENING_STRING + key + UNLIKELY_CLOSING_STRING;
    }
    /**
     * Removing surrounding quotes from any object placeholders
     *
     * Those have been put there by JSON.stringify(), but we need to
     * remove them.
     */
    unquoteKeyPlaceholders(sub) {
        if (this.inputType !== InputType.Object) {
            return sub;
        }
        return core_1.Lazy.stringValue({ produce: (ctx) => core_1.Token.asString(deepUnquote(ctx.resolve(sub))) });
        function deepUnquote(resolved) {
            if (Array.isArray(resolved)) {
                return resolved.map(deepUnquote);
            }
            else if (typeof (resolved) === 'object' && resolved !== null) {
                for (const [key, value] of Object.entries(resolved)) {
                    resolved[key] = deepUnquote(value);
                }
                return resolved;
            }
            else if (typeof (resolved) === 'string') {
                return resolved.replace(OPENING_STRING_REGEX, '<').replace(CLOSING_STRING_REGEX, '>');
            }
            return resolved;
        }
    }
}
const UNLIKELY_OPENING_STRING = '<<${';
const UNLIKELY_CLOSING_STRING = '}>>';
const OPENING_STRING_REGEX = new RegExp(regexQuote('"' + UNLIKELY_OPENING_STRING), 'g');
const CLOSING_STRING_REGEX = new RegExp(regexQuote(UNLIKELY_CLOSING_STRING + '"'), 'g');
/**
 * Represents a field in the event pattern
 */
class EventField {
    /**
     *
     * @param path the path to a field in the event pattern
     */
    constructor(path) {
        this.path = path;
        this.displayHint = this.path.replace(/^[^a-zA-Z0-9_-]+/, '').replace(/[^a-zA-Z0-9_-]/g, '-');
        Object.defineProperty(this, EVENT_FIELD_SYMBOL, { value: true });
        this.creationStack = core_1.captureStackTrace();
    }
    /**
     * Extract the event ID from the event
     */
    static get eventId() {
        return this.fromPath('$.id');
    }
    /**
     * Extract the detail type from the event
     */
    static get detailType() {
        return this.fromPath('$.detail-type');
    }
    /**
     * Extract the source from the event
     */
    static get source() {
        return this.fromPath('$.source');
    }
    /**
     * Extract the account from the event
     */
    static get account() {
        return this.fromPath('$.account');
    }
    /**
     * Extract the time from the event
     */
    static get time() {
        return this.fromPath('$.time');
    }
    /**
     * Extract the region from the event
     */
    static get region() {
        return this.fromPath('$.region');
    }
    /**
     * Extract a custom JSON path from the event
     */
    static fromPath(path) {
        return new EventField(path).toString();
    }
    resolve(_ctx) {
        return this.path;
    }
    toString() {
        return core_1.Token.asString(this, { displayHint: this.displayHint });
    }
    /**
     * Convert the path to the field in the event pattern to JSON
     */
    toJSON() {
        return `<path:${this.path}>`;
    }
}
exports.EventField = EventField;
var InputType;
(function (InputType) {
    InputType[InputType["Object"] = 0] = "Object";
    InputType[InputType["Text"] = 1] = "Text";
    InputType[InputType["Multiline"] = 2] = "Multiline";
})(InputType || (InputType = {}));
function isEventField(x) {
    return EVENT_FIELD_SYMBOL in x;
}
const EVENT_FIELD_SYMBOL = Symbol.for('@aws-cdk/aws-events.EventField');
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, '\\$&');
}
//# sourceMappingURL=data:application/json;base64,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