"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const enums_1 = require("./enums");
/**
 * Render an attribute dict to a list of { key, value } pairs
 */
function renderAttributes(attributes) {
    const ret = [];
    for (const [key, value] of Object.entries(attributes)) {
        if (value !== undefined) {
            ret.push({ key, value });
        }
    }
    return ret;
}
exports.renderAttributes = renderAttributes;
/**
 * Return the appropriate default port for a given protocol
 */
function defaultPortForProtocol(proto) {
    switch (proto) {
        case enums_1.ApplicationProtocol.HTTP: return 80;
        case enums_1.ApplicationProtocol.HTTPS: return 443;
        default:
            throw new Error(`Unrecognized protocol: ${proto}`);
    }
}
exports.defaultPortForProtocol = defaultPortForProtocol;
/**
 * Return the appropriate default protocol for a given port
 */
function defaultProtocolForPort(port) {
    switch (port) {
        case 80:
        case 8000:
        case 8008:
        case 8080:
            return enums_1.ApplicationProtocol.HTTP;
        case 443:
        case 8443:
            return enums_1.ApplicationProtocol.HTTPS;
        default:
            throw new Error(`Don't know default protocol for port: ${port}; please supply a protocol`);
    }
}
exports.defaultProtocolForPort = defaultProtocolForPort;
/**
 * Given a protocol and a port, try to guess the other one if it's undefined
 */
// tslint:disable-next-line:max-line-length
function determineProtocolAndPort(protocol, port) {
    if (protocol === undefined && port === undefined) {
        return [undefined, undefined];
    }
    if (protocol === undefined) {
        protocol = defaultProtocolForPort(port);
    }
    if (port === undefined) {
        port = defaultPortForProtocol(protocol);
    }
    return [protocol, port];
}
exports.determineProtocolAndPort = determineProtocolAndPort;
/**
 * Helper function to default undefined input props
 */
function ifUndefined(x, def) {
    return x !== undefined ? x : def;
}
exports.ifUndefined = ifUndefined;
//# sourceMappingURL=data:application/json;base64,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