"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * ListenerCondition providers definition.
 */
class ListenerCondition {
    /**
     * Create a host-header listener rule condition
     *
     * @param values Hosts for host headers
     */
    static hostHeaders(values) {
        return new HostHeaderListenerCondition(values);
    }
    /**
     * Create a http-header listener rule condition
     *
     * @param name HTTP header name
     * @param values HTTP header values
     */
    static httpHeader(name, values) {
        return new HttpHeaderListenerCondition(name, values);
    }
    /**
     * Create a http-request-method listener rule condition
     *
     * @param values HTTP request methods
     */
    static httpRequestMethods(values) {
        return new HttpRequestMethodListenerCondition(values);
    }
    /**
     * Create a path-pattern listener rule condition
     *
     * @param values Path patterns
     */
    static pathPatterns(values) {
        return new PathPatternListenerCondition(values);
    }
    /**
     * Create a query-string listener rule condition
     *
     * @param values Query string key/value pairs
     */
    static queryStrings(values) {
        return new QueryStringListenerCondition(values);
    }
    /**
     * Create a source-ip listener rule condition
     *
     * @param values Source ips
     */
    static sourceIps(values) {
        return new SourceIpListenerCondition(values);
    }
}
exports.ListenerCondition = ListenerCondition;
/**
 * Host header config of the listener rule condition
 */
class HostHeaderListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'host-header',
            hostHeaderConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * HTTP header config of the listener rule condition
 */
class HttpHeaderListenerCondition extends ListenerCondition {
    constructor(name, values) {
        super();
        this.name = name;
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'http-header',
            httpHeaderConfig: {
                httpHeaderName: this.name,
                values: this.values,
            },
        };
    }
}
/**
 * HTTP reqeust method config of the listener rule condition
 */
class HttpRequestMethodListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'http-request-method',
            httpRequestMethodConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Path pattern config of the listener rule condition
 */
class PathPatternListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'path-pattern',
            pathPatternConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Query string config of the listener rule condition
 */
class QueryStringListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'query-string',
            queryStringConfig: {
                values: this.values,
            },
        };
    }
}
/**
 * Source ip config of the listener rule condition
 */
class SourceIpListenerCondition extends ListenerCondition {
    constructor(values) {
        super();
        this.values = values;
    }
    renderRawCondition() {
        return {
            field: 'source-ip',
            sourceIpConfig: {
                values: this.values,
            },
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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