"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
/**
 * An IP address that is a target for load balancing.
 *
 * Specify IP addresses from the subnets of the virtual private cloud (VPC) for
 * the target group, the RFC 1918 range (10.0.0.0/8, 172.16.0.0/12, and
 * 192.168.0.0/16), and the RFC 6598 range (100.64.0.0/10). You can't specify
 * publicly routable IP addresses.
 *
 * If you register a target of this type, you are responsible for making
 * sure the load balancer's security group can send packets to the IP address.
 */
class IpTarget {
    /**
     * Create a new IPAddress target
     *
     * The availabilityZone parameter determines whether the target receives
     * traffic from the load balancer nodes in the specified Availability Zone
     * or from all enabled Availability Zones for the load balancer.
     *
     * This parameter is not supported if the target type of the target group
     * is instance. If the IP address is in a subnet of the VPC for the target
     * group, the Availability Zone is automatically detected and this
     * parameter is optional. If the IP address is outside the VPC, this
     * parameter is required.
     *
     * With an Application Load Balancer, if the IP address is outside the VPC
     * for the target group, the only supported value is all.
     *
     * Default is automatic.
     *
     * @param ipAddress The IP Address to load balance to
     * @param port Override the group's default port
     * @param availabilityZone Availability zone to send traffic from
     */
    constructor(ipAddress, port, availabilityZone) {
        this.ipAddress = ipAddress;
        this.port = port;
        this.availabilityZone = availabilityZone;
    }
    /**
     * Register this instance target with a load balancer
     *
     * Don't call this, it is called automatically when you add the target to a
     * load balancer.
     */
    attachToApplicationTargetGroup(targetGroup) {
        return this.attach(targetGroup);
    }
    /**
     * Register this instance target with a load balancer
     *
     * Don't call this, it is called automatically when you add the target to a
     * load balancer.
     */
    attachToNetworkTargetGroup(targetGroup) {
        return this.attach(targetGroup);
    }
    attach(_targetGroup) {
        return {
            targetType: elbv2.TargetType.IP,
            targetJson: { id: this.ipAddress, port: this.port, availabilityZone: this.availabilityZone },
        };
    }
}
exports.IpTarget = IpTarget;
//# sourceMappingURL=data:application/json;base64,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