"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.anyValue({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (props.crossZone === undefined || props.crossZone) ? true : false,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: listener.sslCertificateId,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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