"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cloudformation_1 = require("../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const path = require("path");
const kubectl_layer_1 = require("./kubectl-layer");
/**
 * Represents a helm chart within the Kubernetes system.
 *
 * Applies/deletes the resources using `kubectl` in sync with the resource.
 */
class HelmChart extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stack = core_1.Stack.of(this);
        // we maintain a single manifest custom resource handler for each cluster
        const handler = this.getOrCreateHelmChartHandler(props.cluster);
        if (!handler) {
            throw new Error('Cannot define a Helm chart on a cluster with kubectl disabled');
        }
        new aws_cloudformation_1.CustomResource(this, 'Resource', {
            provider: aws_cloudformation_1.CustomResourceProvider.lambda(handler),
            resourceType: HelmChart.RESOURCE_TYPE,
            properties: {
                Release: props.release || this.node.uniqueId.slice(-63).toLowerCase(),
                Chart: props.chart,
                Version: props.version,
                Values: (props.values ? stack.toJsonString(props.values) : undefined),
                Namespace: props.namespace || 'default',
                Repository: props.repository,
            },
        });
    }
    getOrCreateHelmChartHandler(cluster) {
        if (!cluster.kubectlEnabled) {
            return undefined;
        }
        let handler = cluster.node.tryFindChild('HelmChartHandler');
        if (!handler) {
            handler = new lambda.Function(cluster, 'HelmChartHandler', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'helm-chart')),
                runtime: lambda.Runtime.PYTHON_3_7,
                handler: 'index.handler',
                timeout: core_1.Duration.minutes(15),
                layers: [kubectl_layer_1.KubectlLayer.getOrCreate(this, { version: '2.0.0-beta1' })],
                memorySize: 256,
                environment: {
                    CLUSTER_NAME: cluster.clusterName,
                },
                // NOTE: we must use the default IAM role that's mapped to "system:masters"
                // as the execution role of this custom resource handler. This is the only
                // way to be able to interact with the cluster after it's been created.
                role: cluster._defaultMastersRole,
            });
        }
        return handler;
    }
}
exports.HelmChart = HelmChart;
/**
 * The CloudFormation reosurce type.
 */
HelmChart.RESOURCE_TYPE = 'Custom::AWSCDK-EKS-HelmChart';
//# sourceMappingURL=data:application/json;base64,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