"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Linux-specific options that are applied to the container.
 */
class LinuxParameters extends cdk.Construct {
    /**
     * Constructs a new instance of the LinuxParameters class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * Capabilities to be added
         */
        this.capAdd = new Array();
        /**
         * Capabilities to be dropped
         */
        this.capDrop = new Array();
        /**
         * Device mounts
         */
        this.devices = new Array();
        /**
         * TmpFs mounts
         */
        this.tmpfs = new Array();
        this.sharedMemorySize = props.sharedMemorySize;
        this.initProcessEnabled = props.initProcessEnabled;
    }
    /**
     * Adds one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    addCapabilities(...cap) {
        this.capAdd.push(...cap);
    }
    /**
     * Removes one or more Linux capabilities to the Docker configuration of a container.
     *
     * Only works with EC2 launch type.
     */
    dropCapabilities(...cap) {
        this.capDrop.push(...cap);
    }
    /**
     * Adds one or more host devices to a container.
     */
    addDevices(...device) {
        this.devices.push(...device);
    }
    /**
     * Specifies the container path, mount options, and size (in MiB) of the tmpfs mount for a container.
     *
     * Only works with EC2 launch type.
     */
    addTmpfs(...tmpfs) {
        this.tmpfs.push(...tmpfs);
    }
    /**
     * Renders the Linux parameters to a CloudFormation object.
     */
    renderLinuxParameters() {
        return {
            initProcessEnabled: this.initProcessEnabled,
            sharedMemorySize: this.sharedMemorySize,
            capabilities: {
                add: cdk.Lazy.listValue({ produce: () => this.capAdd }, { omitEmpty: true }),
                drop: cdk.Lazy.listValue({ produce: () => this.capDrop }, { omitEmpty: true }),
            },
            devices: cdk.Lazy.anyValue({ produce: () => this.devices.map(renderDevice) }, { omitEmptyArray: true }),
            tmpfs: cdk.Lazy.anyValue({ produce: () => this.tmpfs.map(renderTmpfs) }, { omitEmptyArray: true }),
        };
    }
}
exports.LinuxParameters = LinuxParameters;
function renderDevice(device) {
    return {
        containerPath: device.containerPath,
        hostPath: device.hostPath,
        permissions: device.permissions,
    };
}
function renderTmpfs(tmpfs) {
    return {
        containerPath: tmpfs.containerPath,
        size: tmpfs.size,
        mountOptions: tmpfs.mountOptions,
    };
}
/**
 * A Linux capability
 */
var Capability;
(function (Capability) {
    Capability["ALL"] = "ALL";
    Capability["AUDIT_CONTROL"] = "AUDIT_CONTROL";
    Capability["AUDIT_WRITE"] = "AUDIT_WRITE";
    Capability["BLOCK_SUSPEND"] = "BLOCK_SUSPEND";
    Capability["CHOWN"] = "CHOWN";
    Capability["DAC_OVERRIDE"] = "DAC_OVERRIDE";
    Capability["DAC_READ_SEARCH"] = "DAC_READ_SEARCH";
    Capability["FOWNER"] = "FOWNER";
    Capability["FSETID"] = "FSETID";
    Capability["IPC_LOCK"] = "IPC_LOCK";
    Capability["IPC_OWNER"] = "IPC_OWNER";
    Capability["KILL"] = "KILL";
    Capability["LEASE"] = "LEASE";
    Capability["LINUX_IMMUTABLE"] = "LINUX_IMMUTABLE";
    Capability["MAC_ADMIN"] = "MAC_ADMIN";
    Capability["MAC_OVERRIDE"] = "MAC_OVERRIDE";
    Capability["MKNOD"] = "MKNOD";
    Capability["NET_ADMIN"] = "NET_ADMIN";
    Capability["NET_BIND_SERVICE"] = "NET_BIND_SERVICE";
    Capability["NET_BROADCAST"] = "NET_BROADCAST";
    Capability["NET_RAW"] = "NET_RAW";
    Capability["SETFCAP"] = "SETFCAP";
    Capability["SETGID"] = "SETGID";
    Capability["SETPCAP"] = "SETPCAP";
    Capability["SETUID"] = "SETUID";
    Capability["SYS_ADMIN"] = "SYS_ADMIN";
    Capability["SYS_BOOT"] = "SYS_BOOT";
    Capability["SYS_CHROOT"] = "SYS_CHROOT";
    Capability["SYS_MODULE"] = "SYS_MODULE";
    Capability["SYS_NICE"] = "SYS_NICE";
    Capability["SYS_PACCT"] = "SYS_PACCT";
    Capability["SYS_PTRACE"] = "SYS_PTRACE";
    Capability["SYS_RAWIO"] = "SYS_RAWIO";
    Capability["SYS_RESOURCE"] = "SYS_RESOURCE";
    Capability["SYS_TIME"] = "SYS_TIME";
    Capability["SYS_TTY_CONFIG"] = "SYS_TTY_CONFIG";
    Capability["SYSLOG"] = "SYSLOG";
    Capability["WAKE_ALARM"] = "WAKE_ALARM";
})(Capability = exports.Capability || (exports.Capability = {}));
/**
 * Permissions for device access
 */
var DevicePermission;
(function (DevicePermission) {
    /**
     * Read
     */
    DevicePermission["READ"] = "read";
    /**
     * Write
     */
    DevicePermission["WRITE"] = "write";
    /**
     * Make a node
     */
    DevicePermission["MKNOD"] = "mknod";
})(DevicePermission = exports.DevicePermission || (exports.DevicePermission = {}));
/**
 * The supported options for a tmpfs mount for a container.
 */
var TmpfsMountOption;
(function (TmpfsMountOption) {
    TmpfsMountOption["DEFAULTS"] = "defaults";
    TmpfsMountOption["RO"] = "ro";
    TmpfsMountOption["RW"] = "rw";
    TmpfsMountOption["SUID"] = "suid";
    TmpfsMountOption["NOSUID"] = "nosuid";
    TmpfsMountOption["DEV"] = "dev";
    TmpfsMountOption["NODEV"] = "nodev";
    TmpfsMountOption["EXEC"] = "exec";
    TmpfsMountOption["NOEXEC"] = "noexec";
    TmpfsMountOption["SYNC"] = "sync";
    TmpfsMountOption["ASYNC"] = "async";
    TmpfsMountOption["DIRSYNC"] = "dirsync";
    TmpfsMountOption["REMOUNT"] = "remount";
    TmpfsMountOption["MAND"] = "mand";
    TmpfsMountOption["NOMAND"] = "nomand";
    TmpfsMountOption["ATIME"] = "atime";
    TmpfsMountOption["NOATIME"] = "noatime";
    TmpfsMountOption["DIRATIME"] = "diratime";
    TmpfsMountOption["NODIRATIME"] = "nodiratime";
    TmpfsMountOption["BIND"] = "bind";
    TmpfsMountOption["RBIND"] = "rbind";
    TmpfsMountOption["UNBINDABLE"] = "unbindable";
    TmpfsMountOption["RUNBINDABLE"] = "runbindable";
    TmpfsMountOption["PRIVATE"] = "private";
    TmpfsMountOption["RPRIVATE"] = "rprivate";
    TmpfsMountOption["SHARED"] = "shared";
    TmpfsMountOption["RSHARED"] = "rshared";
    TmpfsMountOption["SLAVE"] = "slave";
    TmpfsMountOption["RSLAVE"] = "rslave";
    TmpfsMountOption["RELATIME"] = "relatime";
    TmpfsMountOption["NORELATIME"] = "norelatime";
    TmpfsMountOption["STRICTATIME"] = "strictatime";
    TmpfsMountOption["NOSTRICTATIME"] = "nostrictatime";
    TmpfsMountOption["MODE"] = "mode";
    TmpfsMountOption["UID"] = "uid";
    TmpfsMountOption["GID"] = "gid";
    TmpfsMountOption["NR_INODES"] = "nr_inodes";
    TmpfsMountOption["NR_BLOCKS"] = "nr_blocks";
    TmpfsMountOption["MPOL"] = "mpol";
})(TmpfsMountOption = exports.TmpfsMountOption || (exports.TmpfsMountOption = {}));
//# sourceMappingURL=data:application/json;base64,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