"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("./container-definition");
const container_image_1 = require("./container-image");
/**
 * Firelens log router type, fluentbit or fluentd.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html
 */
var FirelensLogRouterType;
(function (FirelensLogRouterType) {
    /**
     * fluentbit
     */
    FirelensLogRouterType["FLUENTBIT"] = "fluentbit";
    /**
     * fluentd
     */
    FirelensLogRouterType["FLUENTD"] = "fluentd";
})(FirelensLogRouterType = exports.FirelensLogRouterType || (exports.FirelensLogRouterType = {}));
/**
 * Firelens configuration file type, s3 or file path.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-taskdef-customconfig
 */
var FirelensConfigFileType;
(function (FirelensConfigFileType) {
    /**
     * s3
     */
    FirelensConfigFileType["S3"] = "s3";
    /**
     * fluentd
     */
    FirelensConfigFileType["FILE"] = "file";
})(FirelensConfigFileType = exports.FirelensConfigFileType || (exports.FirelensConfigFileType = {}));
/**
 * Render to CfnTaskDefinition.FirelensConfigurationProperty from FirelensConfig
 */
function renderFirelensConfig(firelensConfig) {
    if (!firelensConfig.options) {
        return { type: firelensConfig.type };
    }
    else {
        // firelensConfig.options.configFileType has been filled with s3 or file type in constructor.
        return {
            type: firelensConfig.type,
            options: {
                'enable-ecs-log-metadata': firelensConfig.options.enableECSLogMetadata ? 'true' : 'false',
                'config-file-type': firelensConfig.options.configFileType,
                'config-file-value': firelensConfig.options.configFileValue,
            },
        };
    }
}
/**
 * SSM parameters for latest fluent bit docker image in ECR
 * https://github.com/aws/aws-for-fluent-bit#using-ssm-to-find-available-versions
 */
const fluentBitImageSSMPath = '/aws/service/aws-for-fluent-bit';
/**
 * Obtain Fluent Bit image in Amazon ECR and setup corresponding IAM permissions.
 * ECR image pull permissions will be granted in task execution role.
 * Cloudwatch logs, Kinesis data stream or firehose permissions will be grant by check options in logDriverConfig.
 * https://docs.aws.amazon.com/AmazonECS/latest/developerguide/using_firelens.html#firelens-using-fluentbit
 */
function obtainDefaultFluentBitECRImage(task, logDriverConfig, imageTag) {
    // grant ECR image pull permissions to executor role
    task.addToExecutionRolePolicy(new iam.PolicyStatement({
        actions: [
            'ecr:GetAuthorizationToken',
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:BatchGetImage',
        ],
        resources: ['*'],
    }));
    // grant cloudwatch or firehose permissions to task role
    const logName = logDriverConfig && logDriverConfig.logDriver === 'awsfirelens'
        && logDriverConfig.options && logDriverConfig.options.Name;
    if (logName === 'cloudwatch') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogGroup',
                'logs:CreateLogStream',
                'logs:DescribeLogStreams',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'firehose') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'firehose:PutRecordBatch',
            ],
            resources: ['*'],
        }));
    }
    else if (logName === 'kinesis') {
        task.addToTaskRolePolicy(new iam.PolicyStatement({
            actions: [
                'kinesis:PutRecords',
            ],
            resources: ['*'],
        }));
    }
    const fluentBitImageTag = imageTag || 'latest';
    const fluentBitImage = `${fluentBitImageSSMPath}/${fluentBitImageTag}`;
    // Not use ContainerImage.fromEcrRepository since it's not support parsing ECR repo URI,
    // use repo ARN might result in complex Fn:: functions in cloudformation template.
    return container_image_1.ContainerImage.fromRegistry(ssm.StringParameter.valueForStringParameter(task, fluentBitImage));
}
exports.obtainDefaultFluentBitECRImage = obtainDefaultFluentBitECRImage;
/**
 * Firelens log router
 */
class FirelensLogRouter extends container_definition_1.ContainerDefinition {
    /**
     * Constructs a new instance of the FirelensLogRouter class.
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const options = props.firelensConfig.options;
        if (options) {
            const enableECSLogMetadata = options.enableECSLogMetadata || options.enableECSLogMetadata === undefined;
            const configFileType = (options.configFileType === undefined || options.configFileType === FirelensConfigFileType.S3) &&
                (cdk.Token.isUnresolved(options.configFileValue) || /arn:aws[a-zA-Z-]*:s3:::.+/.test(options.configFileValue))
                ? FirelensConfigFileType.S3 : FirelensConfigFileType.FILE;
            this.firelensConfig = {
                type: props.firelensConfig.type,
                options: {
                    enableECSLogMetadata,
                    configFileType,
                    configFileValue: options.configFileValue,
                },
            };
            // grant s3 access permissions
            if (configFileType === FirelensConfigFileType.S3) {
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetObject',
                    ],
                    resources: [options.configFileValue],
                }));
                props.taskDefinition.addToExecutionRolePolicy(new iam.PolicyStatement({
                    actions: [
                        's3:GetBucketLocation',
                    ],
                    resources: [options.configFileValue.split('/')[0]],
                }));
            }
        }
        else {
            this.firelensConfig = props.firelensConfig;
        }
    }
    /**
     * Render this container definition to a CloudFormation object
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            ...(super.renderContainerDefinition()),
            firelensConfiguration: this.firelensConfig && renderFirelensConfig(this.firelensConfig),
        };
    }
}
exports.FirelensLogRouter = FirelensLogRouter;
//# sourceMappingURL=data:application/json;base64,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