"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Constructs for types of container images
 */
class ContainerImage {
    /**
     * Reference an image on DockerHub or another online registry
     */
    static fromRegistry(name, props = {}) {
        return new repository_1.RepositoryImage(name, props);
    }
    /**
     * Reference an image in an ECR repository
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new ecr_1.EcrImage(repository, tag);
    }
    /**
     * Reference an image that's constructed directly from sources on disk.
     *
     * If you already have a `DockerImageAsset` instance, you can use the
     * `ContainerImage.fromDockerImageAsset` method instead.
     *
     * @param directory The directory containing the Dockerfile
     */
    static fromAsset(directory, props = {}) {
        return new asset_image_1.AssetImage(directory, props);
    }
    /**
     * Use an existing `DockerImageAsset` for this container image.
     *
     * @param asset The `DockerImageAsset` to use for this container definition.
     */
    static fromDockerImageAsset(asset) {
        return {
            bind(_scope, containerDefinition) {
                asset.repository.grantPull(containerDefinition.taskDefinition.obtainExecutionRole());
                return {
                    imageName: asset.imageUri,
                };
            },
        };
    }
}
exports.ContainerImage = ContainerImage;
const asset_image_1 = require("./images/asset-image");
const ecr_1 = require("./images/ecr");
const repository_1 = require("./images/repository");
//# sourceMappingURL=data:application/json;base64,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