"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
const firelens_log_router_1 = require("../firelens-log-router");
const aws_log_driver_1 = require("../log-drivers/aws-log-driver");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * Return true if the task definition can be run on an EC2 cluster
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * Return true if the task definition can be run on a Fargate cluster
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
}
/**
 * The base class for all task definitions.
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * Constructs a new instance of the TaskDefinition class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The container definitions.
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        this.family = props.family || this.node.uniqueId;
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = props.networkMode !== undefined ? props.networkMode :
            this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE;
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.anyValue({ produce: () => this.renderContainers() }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.anyValue({ produce: () => this.volumes }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.stringValue({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ['EC2'] : []),
                ...(isFargateCompatible(props.compatibility) ? ['FARGATE'] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.anyValue({ produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined,
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
            ipcMode: props.ipcMode,
            pidMode: props.pidMode,
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        class Import extends TaskDefinitionBase {
            constructor() {
                super(...arguments);
                this.taskDefinitionArn = taskDefinitionArn;
                this.compatibility = Compatibility.EC2_AND_FARGATE;
                this.executionRole = undefined;
            }
        }
        return new Import(scope, id);
    }
    get executionRole() {
        return this._executionRole;
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer.findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMappings()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping,
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * Adds a policy statement to the task IAM role.
     */
    addToTaskRolePolicy(statement) {
        this.taskRole.addToPolicy(statement);
    }
    /**
     * Adds a policy statement to the task execution IAM role.
     */
    addToExecutionRolePolicy(statement) {
        this.obtainExecutionRole().addToPolicy(statement);
    }
    /**
     * Adds a new container to the task definition.
     */
    addContainer(id, props) {
        return new container_definition_1.ContainerDefinition(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Adds a firelens log router to the task definition.
     */
    addFirelensLogRouter(id, props) {
        // only one firelens log router is allowed in each task.
        if (this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
            throw new Error('Firelens log router is already added in this task.');
        }
        return new firelens_log_router_1.FirelensLogRouter(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
    }
    /**
     * Adds a volume to the task definition.
     */
    addVolume(volume) {
        this.volumes.push(volume);
    }
    /**
     * Adds the specified placement constraint to the task definition.
     */
    addPlacementConstraint(constraint) {
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * Adds the specified extention to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     */
    addExtension(extension) {
        extension.extend(this);
    }
    /**
     * Creates the task execution IAM role if it doesn't already exist.
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
        }
        return this._executionRole;
    }
    /**
     * Validates the task definition.
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
    renderContainers() {
        // add firelens log router container if any application container is using firelens log driver,
        // also check if already created log router container
        for (const container of this.containers) {
            if (container.logDriverConfig && container.logDriverConfig.logDriver === 'awsfirelens'
                && !this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
                this.addFirelensLogRouter('log-router', {
                    image: firelens_log_router_1.obtainDefaultFluentBitECRImage(this, container.logDriverConfig),
                    firelensConfig: {
                        type: firelens_log_router_1.FirelensLogRouterType.FLUENTBIT,
                    },
                    logging: new aws_log_driver_1.AwsLogDriver({ streamPrefix: 'firelens' }),
                    memoryReservationMiB: 50,
                });
                break;
            }
        }
        return this.containers.map(x => x.renderContainerDefinition());
    }
}
exports.TaskDefinition = TaskDefinition;
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * The networking mode to use for the containers in the task.
 */
var NetworkMode;
(function (NetworkMode) {
    /**
     * The task's containers do not have external connectivity and port mappings can't be specified in the container definition.
     */
    NetworkMode["NONE"] = "none";
    /**
     * The task utilizes Docker's built-in virtual network which runs inside each container instance.
     */
    NetworkMode["BRIDGE"] = "bridge";
    /**
     * The task is allocated an elastic network interface.
     */
    NetworkMode["AWS_VPC"] = "awsvpc";
    /**
     * The task bypasses Docker's built-in virtual network and maps container ports directly to the EC2 instance's network interface directly.
     *
     * In this mode, you can't run multiple instantiations of the same task on a
     * single container instance when port mappings are used.
     */
    NetworkMode["HOST"] = "host";
    /**
     * The task utilizes NAT network mode required by Windows containers.
     *
     * This is the only supported network mode for Windows containers. For more information, see
     * [Task Definition Parameters](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task_definition_parameters.html#network_mode).
     */
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * The IPC resource namespace to use for the containers in the task.
 */
var IpcMode;
(function (IpcMode) {
    /**
     * If none is specified, then IPC resources within the containers of a task are private and not
     * shared with other containers in a task or on the container instance
     */
    IpcMode["NONE"] = "none";
    /**
     * If host is specified, then all containers within the tasks that specified the host IPC mode on
     * the same container instance share the same IPC resources with the host Amazon EC2 instance.
     */
    IpcMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same IPC resources.
     */
    IpcMode["TASK"] = "task";
})(IpcMode = exports.IpcMode || (exports.IpcMode = {}));
/**
 * The process namespace to use for the containers in the task.
 */
var PidMode;
(function (PidMode) {
    /**
     * If host is specified, then all containers within the tasks that specified the host PID mode
     * on the same container instance share the same process namespace with the host Amazon EC2 instance.
     */
    PidMode["HOST"] = "host";
    /**
     * If task is specified, all containers within the specified task share the same process namespace.
     */
    PidMode["TASK"] = "task";
})(PidMode = exports.PidMode || (exports.PidMode = {}));
/**
 * The scope for the Docker volume that determines its lifecycle.
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 */
var Scope;
(function (Scope) {
    /**
     * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
     */
    Scope["TASK"] = "task";
    /**
     * Docker volumes that are scoped as shared persist after the task stops.
     */
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * The task launch type compatibility requirement.
 */
var Compatibility;
(function (Compatibility) {
    /**
     * The task should specify the EC2 launch type.
     */
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    /**
     * The task should specify the Fargate launch type.
     */
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    /**
     * The task can specify either the EC2 or Fargate launch types.
     */
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
//# sourceMappingURL=data:application/json;base64,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