"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const appscaling = require("../../../aws-applicationautoscaling"); // Automatically re-written from '@aws-cdk/aws-applicationautoscaling'
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cloudmap = require("../../../aws-servicediscovery"); // Automatically re-written from '@aws-cdk/aws-servicediscovery'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_definition_1 = require("../base/task-definition");
const ecs_generated_1 = require("../ecs.generated");
const scalable_task_count_1 = require("./scalable-task-count");
/**
 * Base class for configuring listener when registering targets.
 */
class ListenerConfig {
    /**
     * Create a config for adding target group to ALB listener.
     */
    static applicationListener(listener, props) {
        return new ApplicationListenerConfig(listener, props);
    }
    /**
     * Create a config for adding target group to NLB listener.
     */
    static networkListener(listener, props) {
        return new NetworkListenerConfig(listener, props);
    }
}
exports.ListenerConfig = ListenerConfig;
/**
 * Class for configuring application load balancer listener when registering targets.
 */
class ApplicationListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const props = this.props || {};
        const protocol = props.protocol;
        const port = props.port !== undefined ? props.port : (protocol === undefined ? 80 :
            (protocol === elbv2.ApplicationProtocol.HTTPS ? 443 : 80));
        this.listener.addTargets(id, {
            ...props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * Class for configuring network load balancer listener when registering targets.
 */
class NetworkListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const port = this.props !== undefined ? this.props.port : 80;
        this.listener.addTargets(id, {
            ...this.props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * The base class for Ec2Service and FargateService services.
 */
class BaseService extends core_1.Resource {
    /**
     * Constructs a new instance of the BaseService class.
     */
    constructor(scope, id, props, additionalProps, taskDefinition) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.serviceName,
        });
        /**
         * The security groups which manage the allowed network traffic for the service.
         */
        this.connections = new ec2.Connections();
        /**
         * A list of Elastic Load Balancing load balancer objects, containing the load balancer name, the container
         * name (as it appears in a container definition), and the container port to access from the load balancer.
         */
        this.loadBalancers = new Array();
        /**
         * The details of the service discovery registries to assign to this service.
         * For more information, see Service Discovery.
         */
        this.serviceRegistries = new Array();
        this.taskDefinition = taskDefinition;
        this.resource = new ecs_generated_1.CfnService(this, 'Service', {
            desiredCount: props.desiredCount,
            serviceName: this.physicalName,
            loadBalancers: core_1.Lazy.anyValue({ produce: () => this.loadBalancers }, { omitEmptyArray: true }),
            deploymentConfiguration: {
                maximumPercent: props.maxHealthyPercent || 200,
                minimumHealthyPercent: props.minHealthyPercent === undefined ? 50 : props.minHealthyPercent,
            },
            propagateTags: props.propagateTags === PropagatedTagSource.NONE ? undefined : props.propagateTags,
            enableEcsManagedTags: props.enableECSManagedTags === undefined ? false : props.enableECSManagedTags,
            deploymentController: props.deploymentController,
            launchType: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === DeploymentControllerType.EXTERNAL ? undefined : props.launchType,
            healthCheckGracePeriodSeconds: this.evaluateHealthGracePeriod(props.healthCheckGracePeriod),
            /* role: never specified, supplanted by Service Linked Role */
            networkConfiguration: core_1.Lazy.anyValue({ produce: () => this.networkConfiguration }, { omitEmptyArray: true }),
            serviceRegistries: core_1.Lazy.anyValue({ produce: () => this.serviceRegistries }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (((_b = props.deploymentController) === null || _b === void 0 ? void 0 : _b.type) === DeploymentControllerType.EXTERNAL) {
            this.node.addWarning('taskDefinition and launchType are blanked out when using external deployment controller.');
        }
        this.serviceArn = this.getResourceArnAttribute(this.resource.ref, {
            service: 'ecs',
            resource: 'service',
            resourceName: `${props.cluster.clusterName}/${this.physicalName}`,
        });
        this.serviceName = this.getResourceNameAttribute(this.resource.attrName);
        this.cluster = props.cluster;
        if (props.cloudMapOptions) {
            this.enableCloudMap(props.cloudMapOptions);
        }
    }
    /**
     * The CloudMap service created for this service, if any.
     */
    get cloudMapService() {
        return this.cloudmapService;
    }
    /**
     * This method is called to attach this service to an Application Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToApplicationTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToApplicationTargetGroup(targetGroup);
    }
    /**
     * Registers the service as a target of a Classic Load Balancer (CLB).
     *
     * Don't call this. Call `loadBalancer.addTarget()` instead.
     */
    attachToClassicLB(loadBalancer) {
        return this.defaultLoadBalancerTarget.attachToClassicLB(loadBalancer);
    }
    /**
     * Return a load balancing target for a specific container and port.
     *
     * Use this function to create a load balancer target if you want to load balance to
     * another container than the first essential container or the first mapped port on
     * the container.
     *
     * Use the return value of this function where you would normally use a load balancer
     * target, instead of the `Service` object itself.
     *
     * @example
     *
     * listener.addTargets(service.loadBalancerTarget({
     *   containerName: 'MyContainer',
     *   containerPort: 1234
     * }));
     */
    loadBalancerTarget(options) {
        const self = this;
        const target = this.taskDefinition._validateTarget(options);
        const connections = self.connections;
        return {
            attachToApplicationTargetGroup(targetGroup) {
                targetGroup.registerConnectable(self, self.taskDefinition._portRangeFromPortMapping(target.portMapping));
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            attachToNetworkTargetGroup(targetGroup) {
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            connections,
            attachToClassicLB(loadBalancer) {
                return self.attachToELB(loadBalancer, target.containerName, target.portMapping.containerPort);
            },
        };
    }
    /**
     * Use this function to create all load balancer targets to be registered in this service, add them to
     * target groups, and attach target groups to listeners accordingly.
     *
     * Alternatively, you can use `listener.addTargets()` to create targets and add them to target groups.
     *
     * @example
     *
     * service.registerLoadBalancerTargets(
     *   {
     *     containerName: 'web',
     *     containerPort: 80,
     *     newTargetGroupId: 'ECS',
     *     listener: ecs.ListenerConfig.applicationListener(listener, {
     *       protocol: elbv2.ApplicationProtocol.HTTPS
     *     }),
     *   },
     * )
     */
    registerLoadBalancerTargets(...targets) {
        for (const target of targets) {
            target.listener.addTargets(target.newTargetGroupId, {
                containerName: target.containerName,
                containerPort: target.containerPort,
                protocol: target.protocol,
            }, this);
        }
    }
    /**
     * This method is called to attach this service to a Network Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToNetworkTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToNetworkTargetGroup(targetGroup);
    }
    /**
     * An attribute representing the minimum and maximum task count for an AutoScalingGroup.
     */
    autoScaleTaskCount(props) {
        if (this.scalableTaskCount) {
            throw new Error('AutoScaling of task count already enabled for this service');
        }
        return this.scalableTaskCount = new scalable_task_count_1.ScalableTaskCount(this, 'TaskCount', {
            serviceNamespace: appscaling.ServiceNamespace.ECS,
            resourceId: `service/${this.cluster.clusterName}/${this.serviceName}`,
            dimension: 'ecs:service:DesiredCount',
            role: this.makeAutoScalingRole(),
            ...props,
        });
    }
    /**
     * Enable CloudMap service discovery for the service
     *
     * @returns The created CloudMap service
     */
    enableCloudMap(options) {
        const sdNamespace = options.cloudMapNamespace !== undefined ? options.cloudMapNamespace : this.cluster.defaultCloudMapNamespace;
        if (sdNamespace === undefined) {
            throw new Error('Cannot enable service discovery if a Cloudmap Namespace has not been created in the cluster.');
        }
        // Determine DNS type based on network mode
        const networkMode = this.taskDefinition.networkMode;
        if (networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a service discovery if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        // Bridge or host network mode requires SRV records
        let dnsRecordType = options.dnsRecordType;
        if (networkMode === task_definition_1.NetworkMode.BRIDGE || networkMode === task_definition_1.NetworkMode.HOST) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.SRV;
            }
            if (dnsRecordType !== cloudmap.DnsRecordType.SRV) {
                throw new Error('SRV records must be used when network mode is Bridge or Host.');
            }
        }
        // Default DNS record type for AwsVpc network mode is A Records
        if (networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.A;
            }
        }
        // If the task definition that your service task specifies uses the AWSVPC network mode and a type SRV DNS record is
        // used, you must specify a containerName and containerPort combination
        const containerName = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerName : undefined;
        const containerPort = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerPort : undefined;
        const cloudmapService = new cloudmap.Service(this, 'CloudmapService', {
            namespace: sdNamespace,
            name: options.name,
            dnsRecordType: dnsRecordType,
            customHealthCheck: { failureThreshold: options.failureThreshold || 1 },
            dnsTtl: options.dnsTtl,
        });
        const serviceArn = cloudmapService.serviceArn;
        // add Cloudmap service to the ECS Service's serviceRegistry
        this.addServiceRegistry({
            arn: serviceArn,
            containerName,
            containerPort,
        });
        this.cloudmapService = cloudmapService;
        return cloudmapService;
    }
    /**
     * This method returns the specified CloudWatch metric name for this service.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensions: { ClusterName: this.cluster.clusterName, ServiceName: this.serviceName },
            ...props,
        }).attachTo(this);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory utilization.
     *
     * @default average over 5 minutes
     */
    metricMemoryUtilization(props) {
        return this.metric('MemoryUtilization', props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU utilization.
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('CPUUtilization', props);
    }
    /**
     * This method is called to create a networkConfiguration.
     * @deprecated use configureAwsVpcNetworkingWithSecurityGroups instead.
     */
    // tslint:disable-next-line:max-line-length
    configureAwsVpcNetworking(vpc, assignPublicIp, vpcSubnets, securityGroup) {
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroup === undefined) {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', { vpc });
        }
        this.connections.addSecurityGroup(securityGroup);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: core_1.Lazy.listValue({ produce: () => [securityGroup.securityGroupId] }),
            },
        };
    }
    /**
     * This method is called to create a networkConfiguration.
     */
    // tslint:disable-next-line:max-line-length
    configureAwsVpcNetworkingWithSecurityGroups(vpc, assignPublicIp, vpcSubnets, securityGroups) {
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroups === undefined || securityGroups.length === 0) {
            securityGroups = [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc })];
        }
        securityGroups.forEach((sg) => { this.connections.addSecurityGroup(sg); }, this);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: securityGroups.map((sg) => sg.securityGroupId),
            },
        };
    }
    renderServiceRegistry(registry) {
        return {
            registryArn: registry.arn,
            containerName: registry.containerName,
            containerPort: registry.containerPort,
        };
    }
    /**
     * Shared logic for attaching to an ELB
     */
    attachToELB(loadBalancer, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is AwsVpc. Use Host or Bridge instead.');
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is None. Use Host or Bridge instead.');
        }
        this.loadBalancers.push({
            loadBalancerName: loadBalancer.loadBalancerName,
            containerName,
            containerPort,
        });
    }
    /**
     * Shared logic for attaching to an ELBv2
     */
    attachToELBv2(targetGroup, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a load balancer if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        this.loadBalancers.push({
            targetGroupArn: targetGroup.targetGroupArn,
            containerName,
            containerPort,
        });
        // Service creation can only happen after the load balancer has
        // been associated with our target group(s), so add ordering dependency.
        this.resource.node.addDependency(targetGroup.loadBalancerAttached);
        const targetType = this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC ? elbv2.TargetType.IP : elbv2.TargetType.INSTANCE;
        return { targetType };
    }
    get defaultLoadBalancerTarget() {
        return this.loadBalancerTarget({
            containerName: this.taskDefinition.defaultContainer.containerName,
        });
    }
    /**
     * Generate the role that will be used for autoscaling this service
     */
    makeAutoScalingRole() {
        // Use a Service Linked Role.
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            region: '',
            service: 'iam',
            resource: 'role/aws-service-role/ecs.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_ECSService',
        }));
    }
    /**
     * Associate Service Discovery (Cloud Map) service
     */
    addServiceRegistry(registry) {
        const sr = this.renderServiceRegistry(registry);
        this.serviceRegistries.push(sr);
    }
    /**
     *  Return the default grace period when load balancers are configured and
     *  healthCheckGracePeriod is not already set
     */
    evaluateHealthGracePeriod(providedHealthCheckGracePeriod) {
        return core_1.Lazy.anyValue({
            produce: () => providedHealthCheckGracePeriod !== undefined ? providedHealthCheckGracePeriod.toSeconds() :
                this.loadBalancers.length > 0 ? 60 :
                    undefined,
        });
    }
}
exports.BaseService = BaseService;
/**
 * The launch type of an ECS service
 */
var LaunchType;
(function (LaunchType) {
    /**
     * The service will be launched using the EC2 launch type
     */
    LaunchType["EC2"] = "EC2";
    /**
     * The service will be launched using the FARGATE launch type
     */
    LaunchType["FARGATE"] = "FARGATE";
})(LaunchType = exports.LaunchType || (exports.LaunchType = {}));
/**
 * The deployment controller type to use for the service.
 */
var DeploymentControllerType;
(function (DeploymentControllerType) {
    /**
     * The rolling update (ECS) deployment type involves replacing the current
     * running version of the container with the latest version.
     */
    DeploymentControllerType["ECS"] = "ECS";
    /**
     * The blue/green (CODE_DEPLOY) deployment type uses the blue/green deployment model powered by AWS CodeDeploy
     */
    DeploymentControllerType["CODE_DEPLOY"] = "CODE_DEPLOY";
    /**
     * The external (EXTERNAL) deployment type enables you to use any third-party deployment controller
     */
    DeploymentControllerType["EXTERNAL"] = "EXTERNAL";
})(DeploymentControllerType = exports.DeploymentControllerType || (exports.DeploymentControllerType = {}));
/**
 * Propagate tags from either service or task definition
 */
var PropagatedTagSource;
(function (PropagatedTagSource) {
    /**
     * Propagate tags from service
     */
    PropagatedTagSource["SERVICE"] = "SERVICE";
    /**
     * Propagate tags from task definition
     */
    PropagatedTagSource["TASK_DEFINITION"] = "TASK_DEFINITION";
    /**
     * Do not propagate
     */
    PropagatedTagSource["NONE"] = "NONE";
})(PropagatedTagSource = exports.PropagatedTagSource || (exports.PropagatedTagSource = {}));
//# sourceMappingURL=data:application/json;base64,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