"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const net = require("net");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The VPN connection type.
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    /**
     * The IPsec 1 VPN connection type.
     */
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    /**
     * Dummy member
     * TODO: remove once https://github.com/aws/jsii/issues/231 is fixed
     */
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * The VPN Gateway that shall be added to the VPC
 *
 * @resource AWS::EC2::VPNGateway
 */
class VpnGateway extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        // This is 'Default' instead of 'Resource', because using 'Default' will generate
        // a logical ID for a VpnGateway which is exactly the same as the logical ID that used
        // to be created for the CfnVPNGateway (and 'Resource' would not do that).
        const vpnGW = new ec2_generated_1.CfnVPNGateway(this, 'Default', props);
        this.gatewayId = vpnGW.ref;
    }
}
exports.VpnGateway = VpnGateway;
/**
 * Define a VPN Connection
 *
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.vpc.vpnGatewayId) {
            props.vpc.enableVpnGateway({
                type: 'ipsec.1',
                amazonSideAsn: props.asn,
            });
        }
        if (!net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    // tslint:disable:max-line-length
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    // tslint:enable:max-line-length
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        // tslint:disable:max-line-length
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                        // tslint:enable:max-line-length
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: props.tunnelOptions,
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * Return the given named metric for all VPN connections in the account/region.
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,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