"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
/**
 * Peer object factories (to be used in Security Group management)
 *
 * The static methods on this object can be used to create peer objects
 * which represent a connection partner in Security Group rules.
 *
 * Use this object if you need to represent connection partners using plain IP
 * addresses, or a prefix list ID.
 *
 * If you want to address a connection partner by Security Group, you can just
 * use the Security Group (or the construct that contains a Security Group)
 * directly, as it already implements `IPeer`.
 */
class Peer {
    /**
     * Create an IPv4 peer from a CIDR
     */
    static ipv4(cidrIp) {
        return new CidrIPv4(cidrIp);
    }
    /**
     * Any IPv4 address
     */
    static anyIpv4() {
        return new AnyIPv4();
    }
    /**
     * Create an IPv6 peer from a CIDR
     */
    static ipv6(cidrIp) {
        return new CidrIPv6(cidrIp);
    }
    /**
     * Any IPv6 address
     */
    static anyIpv6() {
        return new AnyIPv6();
    }
    /**
     * A prefix list
     */
    static prefixList(prefixListId) {
        return new PrefixList(prefixListId);
    }
    constructor() {
    }
}
exports.Peer = Peer;
/**
 * A connection to and from a given IP range
 */
class CidrIPv4 {
    constructor(cidrIp) {
        this.cidrIp = cidrIp;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIp)) {
            const cidrMatch = cidrIp.match(/^(\d{1,3}\.){3}\d{1,3}(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv4 CIDR: "${cidrIp}"`);
            }
            if (!cidrMatch[2]) {
                throw new Error(`CIDR mask is missing in IPv4: "${cidrIp}". Did you mean "${cidrIp}/32"?`);
            }
        }
        this.uniqueId = cidrIp;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
}
/**
 * Any IPv4 address
 */
class AnyIPv4 extends CidrIPv4 {
    constructor() {
        super('0.0.0.0/0');
    }
}
/**
 * A connection to a from a given IPv6 range
 */
class CidrIPv6 {
    constructor(cidrIpv6) {
        this.cidrIpv6 = cidrIpv6;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIpv6)) {
            const cidrMatch = cidrIpv6.match(/^([\da-f]{0,4}:){2,7}([\da-f]{0,4})?(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv6 CIDR: "${cidrIpv6}"`);
            }
            if (!cidrMatch[3]) {
                throw new Error(`CIDR mask is missing in IPv6: "${cidrIpv6}". Did you mean "${cidrIpv6}/128"?`);
            }
        }
        this.uniqueId = cidrIpv6;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
}
/**
 * Any IPv6 address
 */
class AnyIPv6 extends CidrIPv6 {
    constructor() {
        super('::/0');
    }
}
/**
 * A prefix list
 *
 * Prefix lists are used to allow traffic to VPC-local service endpoints.
 *
 * For more information, see this page:
 *
 * https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-endpoints.html
 */
class PrefixList {
    constructor(prefixListId) {
        this.prefixListId = prefixListId;
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ peer: this });
        this.uniqueId = prefixListId;
    }
    toIngressRuleConfig() {
        return { sourcePrefixListId: this.prefixListId };
    }
    toEgressRuleConfig() {
        return { destinationPrefixListId: this.prefixListId };
    }
}
//# sourceMappingURL=data:application/json;base64,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