"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rule_1 = require("./rule");
/**
 * Checks whether the active access keys are rotated within the number of days
 * specified in `maxAge`.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 *
 * @resource AWS::Config::ConfigRule
 */
class AccessKeysRotated extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'ACCESS_KEYS_ROTATED',
            inputParameters: {
                ...props.maxAge
                    ? {
                        maxAccessKeyAge: props.maxAge.toDays(),
                    }
                    : {},
            },
        });
    }
}
exports.AccessKeysRotated = AccessKeysRotated;
/**
 * Checks whether your CloudFormation stacks' actual configuration differs, or
 * has drifted, from its expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackDriftDetectionCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK',
            inputParameters: {
                cloudformationRoleArn: core_1.Lazy.stringValue({ produce: () => this.role.roleArn }),
            },
        });
        this.scopeToResource('AWS::CloudFormation::Stack', props.ownStackOnly ? core_1.Stack.of(this).stackId : undefined);
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('config.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'),
            ],
        });
    }
}
exports.CloudFormationStackDriftDetectionCheck = CloudFormationStackDriftDetectionCheck;
/**
 * Checks whether your CloudFormation stacks are sending event notifications to
 * a SNS topic. Optionally checks whether specified SNS topics are used.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 *
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackNotificationCheck extends rule_1.ManagedRule {
    constructor(scope, id, props = {}) {
        if (props.topics && props.topics.length > 5) {
            throw new Error('At most 5 topics can be specified.');
        }
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK',
            inputParameters: props.topics && props.topics.reduce((params, topic, idx) => ({ ...params, [`snsTopic${idx + 1}`]: topic.topicArn }), {}),
        });
        this.scopeToResource('AWS::CloudFormation::Stack');
    }
}
exports.CloudFormationStackNotificationCheck = CloudFormationStackNotificationCheck;
//# sourceMappingURL=data:application/json;base64,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