"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cognito_generated_1 = require("./cognito.generated");
const user_pool_client_1 = require("./user-pool-client");
const user_pool_domain_1 = require("./user-pool-domain");
/**
 * User pool operations to which lambda triggers can be attached.
 */
class UserPoolOperation {
    constructor(operationName) {
        this.operationName = operationName;
    }
    /** A custom user pool operation */
    static of(name) {
        const lowerCamelCase = name.charAt(0).toLowerCase() + name.slice(1);
        return new UserPoolOperation(lowerCamelCase);
    }
}
exports.UserPoolOperation = UserPoolOperation;
/**
 * Creates a challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-create-auth-challenge.html
 */
UserPoolOperation.CREATE_AUTH_CHALLENGE = new UserPoolOperation('createAuthChallenge');
/**
 * Advanced customization and localization of messages
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-custom-message.html
 */
UserPoolOperation.CUSTOM_MESSAGE = new UserPoolOperation('customMessage');
/**
 * Determines the next challenge in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-define-auth-challenge.html
 */
UserPoolOperation.DEFINE_AUTH_CHALLENGE = new UserPoolOperation('defineAuthChallenge');
/**
 * Event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-authentication.html
 */
UserPoolOperation.POST_AUTHENTICATION = new UserPoolOperation('postAuthentication');
/**
 * Custom welcome messages or event logging for custom analytics
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-post-confirmation.html
 */
UserPoolOperation.POST_CONFIRMATION = new UserPoolOperation('postConfirmation');
/**
 * Custom validation to accept or deny the sign-in request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-authentication.html
 */
UserPoolOperation.PRE_AUTHENTICATION = new UserPoolOperation('preAuthentication');
/**
 * Custom validation to accept or deny the sign-up request
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-sign-up.html
 */
UserPoolOperation.PRE_SIGN_UP = new UserPoolOperation('preSignUp');
/**
 * Add or remove attributes in Id tokens
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-pre-token-generation.html
 */
UserPoolOperation.PRE_TOKEN_GENERATION = new UserPoolOperation('preTokenGeneration');
/**
 * Migrate a user from an existing user directory to user pools
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-migrate-user.html
 */
UserPoolOperation.USER_MIGRATION = new UserPoolOperation('userMigration');
/**
 * Determines if a response is correct in a custom auth flow
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-lambda-verify-auth-challenge-response.html
 */
UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE = new UserPoolOperation('verifyAuthChallengeResponse');
/**
 * The email verification style
 */
var VerificationEmailStyle;
(function (VerificationEmailStyle) {
    /** Verify email via code */
    VerificationEmailStyle["CODE"] = "CONFIRM_WITH_CODE";
    /** Verify email via link */
    VerificationEmailStyle["LINK"] = "CONFIRM_WITH_LINK";
})(VerificationEmailStyle = exports.VerificationEmailStyle || (exports.VerificationEmailStyle = {}));
/**
 * The different ways in which a user pool's MFA enforcement can be configured.
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-mfa.html
 */
var Mfa;
(function (Mfa) {
    /** Users are not required to use MFA for sign in, and cannot configure one. */
    Mfa["OFF"] = "OFF";
    /** Users are not required to use MFA for sign in, but can configure one if they so choose to. */
    Mfa["OPTIONAL"] = "OPTIONAL";
    /** Users are required to configure an MFA, and have to use it to sign in. */
    Mfa["REQUIRED"] = "ON";
})(Mfa = exports.Mfa || (exports.Mfa = {}));
class UserPoolBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.identityProviders = [];
    }
    addClient(id, options) {
        return new user_pool_client_1.UserPoolClient(this, id, {
            userPool: this,
            ...options,
        });
    }
    addDomain(id, options) {
        return new user_pool_domain_1.UserPoolDomain(this, id, {
            userPool: this,
            ...options,
        });
    }
    registerIdentityProvider(provider) {
        this.identityProviders.push(provider);
    }
}
/**
 * Define a Cognito User Pool
 */
class UserPool extends UserPoolBase {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.triggers = {};
        const signIn = this.signInConfiguration(props);
        if (props.lambdaTriggers) {
            for (const t of Object.keys(props.lambdaTriggers)) {
                const trigger = props.lambdaTriggers[t];
                if (trigger !== undefined) {
                    this.addLambdaPermission(trigger, t);
                    this.triggers[t] = trigger.functionArn;
                }
            }
        }
        const verificationMessageTemplate = this.verificationMessageConfiguration(props);
        let emailVerificationMessage;
        let emailVerificationSubject;
        if (verificationMessageTemplate.defaultEmailOption === VerificationEmailStyle.CODE) {
            emailVerificationMessage = verificationMessageTemplate.emailMessage;
            emailVerificationSubject = verificationMessageTemplate.emailSubject;
        }
        const smsVerificationMessage = verificationMessageTemplate.smsMessage;
        const inviteMessageTemplate = {
            emailMessage: (_a = props.userInvitation) === null || _a === void 0 ? void 0 : _a.emailBody,
            emailSubject: (_b = props.userInvitation) === null || _b === void 0 ? void 0 : _b.emailSubject,
            smsMessage: (_c = props.userInvitation) === null || _c === void 0 ? void 0 : _c.smsMessage,
        };
        const selfSignUpEnabled = props.selfSignUpEnabled !== undefined ? props.selfSignUpEnabled : false;
        const adminCreateUserConfig = {
            allowAdminCreateUserOnly: !selfSignUpEnabled,
            inviteMessageTemplate: props.userInvitation !== undefined ? inviteMessageTemplate : undefined,
        };
        const passwordPolicy = this.configurePasswordPolicy(props);
        const userPool = new cognito_generated_1.CfnUserPool(this, 'Resource', {
            userPoolName: props.userPoolName,
            usernameAttributes: signIn.usernameAttrs,
            aliasAttributes: signIn.aliasAttrs,
            autoVerifiedAttributes: signIn.autoVerifyAttrs,
            lambdaConfig: core_1.Lazy.anyValue({ produce: () => undefinedIfNoKeys(this.triggers) }),
            smsConfiguration: this.smsConfiguration(props),
            adminCreateUserConfig,
            emailVerificationMessage,
            emailVerificationSubject,
            smsVerificationMessage,
            verificationMessageTemplate,
            schema: this.schemaConfiguration(props),
            mfaConfiguration: props.mfa,
            enabledMfas: this.mfaConfiguration(props),
            policies: passwordPolicy !== undefined ? { passwordPolicy } : undefined,
            emailConfiguration: undefinedIfNoKeys({
                from: (_d = props.emailSettings) === null || _d === void 0 ? void 0 : _d.from,
                replyToEmailAddress: (_e = props.emailSettings) === null || _e === void 0 ? void 0 : _e.replyTo,
            }),
            usernameConfiguration: undefinedIfNoKeys({
                caseSensitive: props.signInCaseSensitive,
            }),
        });
        this.userPoolId = userPool.ref;
        this.userPoolArn = userPool.attrArn;
        this.userPoolProviderName = userPool.attrProviderName;
        this.userPoolProviderUrl = userPool.attrProviderUrl;
    }
    /**
     * Import an existing user pool based on its id.
     */
    static fromUserPoolId(scope, id, userPoolId) {
        class Import extends UserPoolBase {
            constructor() {
                super(...arguments);
                this.userPoolId = userPoolId;
                this.userPoolArn = core_1.Stack.of(this).formatArn({
                    service: 'cognito-idp',
                    resource: 'userpool',
                    resourceName: userPoolId,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing user pool based on its ARN.
     */
    static fromUserPoolArn(scope, id, userPoolArn) {
        return UserPool.fromUserPoolId(scope, id, core_1.Stack.of(scope).parseArn(userPoolArn).resourceName);
    }
    /**
     * Add a lambda trigger to a user pool operation
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-identity-pools-working-with-aws-lambda-triggers.html
     */
    addTrigger(operation, fn) {
        if (operation.operationName in this.triggers) {
            throw new Error(`A trigger for the operation ${operation} already exists.`);
        }
        this.addLambdaPermission(fn, operation.operationName);
        this.triggers[operation.operationName] = fn.functionArn;
    }
    addLambdaPermission(fn, name) {
        const capitalize = name.charAt(0).toUpperCase() + name.slice(1);
        fn.addPermission(`${capitalize}Cognito`, {
            principal: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com'),
            sourceArn: this.userPoolArn,
        });
    }
    verificationMessageConfiguration(props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k;
        const CODE_TEMPLATE = '{####}';
        const VERIFY_EMAIL_TEMPLATE = '{##Verify Email##}';
        const emailStyle = (_b = (_a = props.userVerification) === null || _a === void 0 ? void 0 : _a.emailStyle) !== null && _b !== void 0 ? _b : VerificationEmailStyle.CODE;
        const emailSubject = (_d = (_c = props.userVerification) === null || _c === void 0 ? void 0 : _c.emailSubject) !== null && _d !== void 0 ? _d : 'Verify your new account';
        const smsMessage = (_f = (_e = props.userVerification) === null || _e === void 0 ? void 0 : _e.smsMessage) !== null && _f !== void 0 ? _f : `The verification code to your new account is ${CODE_TEMPLATE}`;
        if (emailStyle === VerificationEmailStyle.CODE) {
            const emailMessage = (_h = (_g = props.userVerification) === null || _g === void 0 ? void 0 : _g.emailBody) !== null && _h !== void 0 ? _h : `The verification code to your new account is ${CODE_TEMPLATE}`;
            if (emailMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${CODE_TEMPLATE}'`);
            }
            if (smsMessage.indexOf(CODE_TEMPLATE) < 0) {
                throw new Error(`SMS message must contain the template string '${CODE_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.CODE,
                emailMessage,
                emailSubject,
                smsMessage,
            };
        }
        else {
            const emailMessage = (_k = (_j = props.userVerification) === null || _j === void 0 ? void 0 : _j.emailBody) !== null && _k !== void 0 ? _k : `Verify your account by clicking on ${VERIFY_EMAIL_TEMPLATE}`;
            if (emailMessage.indexOf(VERIFY_EMAIL_TEMPLATE) < 0) {
                throw new Error(`Verification email body must contain the template string '${VERIFY_EMAIL_TEMPLATE}'`);
            }
            return {
                defaultEmailOption: VerificationEmailStyle.LINK,
                emailMessageByLink: emailMessage,
                emailSubjectByLink: emailSubject,
                smsMessage,
            };
        }
    }
    signInConfiguration(props) {
        var _a;
        let aliasAttrs;
        let usernameAttrs;
        let autoVerifyAttrs;
        const signIn = (_a = props.signInAliases) !== null && _a !== void 0 ? _a : { username: true };
        if (signIn.preferredUsername && !signIn.username) {
            throw new Error('username signIn must be enabled if preferredUsername is enabled');
        }
        if (signIn.username) {
            aliasAttrs = [];
            if (signIn.email) {
                aliasAttrs.push("email" /* EMAIL */);
            }
            if (signIn.phone) {
                aliasAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
            if (signIn.preferredUsername) {
                aliasAttrs.push("preferred_username" /* PREFERRED_USERNAME */);
            }
            if (aliasAttrs.length === 0) {
                aliasAttrs = undefined;
            }
        }
        else {
            usernameAttrs = [];
            if (signIn.email) {
                usernameAttrs.push("email" /* EMAIL */);
            }
            if (signIn.phone) {
                usernameAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
        }
        if (props.autoVerify) {
            autoVerifyAttrs = [];
            if (props.autoVerify.email) {
                autoVerifyAttrs.push("email" /* EMAIL */);
            }
            if (props.autoVerify.phone) {
                autoVerifyAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
        }
        else if (signIn.email || signIn.phone) {
            autoVerifyAttrs = [];
            if (signIn.email) {
                autoVerifyAttrs.push("email" /* EMAIL */);
            }
            if (signIn.phone) {
                autoVerifyAttrs.push("phone_number" /* PHONE_NUMBER */);
            }
        }
        return { usernameAttrs, aliasAttrs, autoVerifyAttrs };
    }
    smsConfiguration(props) {
        var _a;
        if (props.smsRole) {
            return {
                snsCallerArn: props.smsRole.roleArn,
                externalId: props.smsRoleExternalId,
            };
        }
        else {
            const smsRoleExternalId = this.node.uniqueId.substr(0, 1223); // sts:ExternalId max length of 1224
            const smsRole = (_a = props.smsRole) !== null && _a !== void 0 ? _a : new aws_iam_1.Role(this, 'smsRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('cognito-idp.amazonaws.com', {
                    conditions: {
                        StringEquals: { 'sts:ExternalId': smsRoleExternalId },
                    },
                }),
                inlinePolicies: {
                    /*
                     * The UserPool is very particular that it must contain an 'sns:Publish' action as an inline policy.
                     * Ideally, a conditional that restricts this action to 'sms' protocol needs to be attached, but the UserPool deployment fails validation.
                     * Seems like a case of being excessively strict.
                     */
                    'sns-publish': new aws_iam_1.PolicyDocument({
                        statements: [
                            new aws_iam_1.PolicyStatement({
                                actions: ['sns:Publish'],
                                resources: ['*'],
                            }),
                        ],
                    }),
                },
            });
            return {
                externalId: smsRoleExternalId,
                snsCallerArn: smsRole.roleArn,
            };
        }
    }
    mfaConfiguration(props) {
        if (props.mfa === undefined || props.mfa === Mfa.OFF) {
            // since default is OFF, treat undefined and OFF the same way
            return undefined;
        }
        else if (props.mfaSecondFactor === undefined &&
            (props.mfa === Mfa.OPTIONAL || props.mfa === Mfa.REQUIRED)) {
            return ['SMS_MFA'];
        }
        else {
            const enabledMfas = [];
            if (props.mfaSecondFactor.sms) {
                enabledMfas.push('SMS_MFA');
            }
            if (props.mfaSecondFactor.otp) {
                enabledMfas.push('SOFTWARE_TOKEN_MFA');
            }
            return enabledMfas;
        }
    }
    configurePasswordPolicy(props) {
        var _a, _b, _c, _d, _e, _f;
        const tempPasswordValidity = (_a = props.passwordPolicy) === null || _a === void 0 ? void 0 : _a.tempPasswordValidity;
        if (tempPasswordValidity !== undefined && tempPasswordValidity.toDays() > core_1.Duration.days(365).toDays()) {
            throw new Error(`tempPasswordValidity cannot be greater than 365 days (received: ${tempPasswordValidity.toDays()})`);
        }
        const minLength = props.passwordPolicy ? (_b = props.passwordPolicy.minLength) !== null && _b !== void 0 ? _b : 8 : undefined;
        if (minLength !== undefined && (minLength < 6 || minLength > 99)) {
            throw new Error(`minLength for password must be between 6 and 99 (received: ${minLength})`);
        }
        return undefinedIfNoKeys({
            temporaryPasswordValidityDays: tempPasswordValidity === null || tempPasswordValidity === void 0 ? void 0 : tempPasswordValidity.toDays({ integral: true }),
            minimumLength: minLength,
            requireLowercase: (_c = props.passwordPolicy) === null || _c === void 0 ? void 0 : _c.requireLowercase,
            requireUppercase: (_d = props.passwordPolicy) === null || _d === void 0 ? void 0 : _d.requireUppercase,
            requireNumbers: (_e = props.passwordPolicy) === null || _e === void 0 ? void 0 : _e.requireDigits,
            requireSymbols: (_f = props.passwordPolicy) === null || _f === void 0 ? void 0 : _f.requireSymbols,
        });
    }
    schemaConfiguration(props) {
        const schema = [];
        if (props.requiredAttributes) {
            const stdAttributes = [];
            if (props.requiredAttributes.address) {
                stdAttributes.push("address" /* ADDRESS */);
            }
            if (props.requiredAttributes.birthdate) {
                stdAttributes.push("birthdate" /* BIRTHDATE */);
            }
            if (props.requiredAttributes.email) {
                stdAttributes.push("email" /* EMAIL */);
            }
            if (props.requiredAttributes.familyName) {
                stdAttributes.push("family_name" /* FAMILY_NAME */);
            }
            if (props.requiredAttributes.fullname) {
                stdAttributes.push("name" /* NAME */);
            }
            if (props.requiredAttributes.gender) {
                stdAttributes.push("gender" /* GENDER */);
            }
            if (props.requiredAttributes.givenName) {
                stdAttributes.push("given_name" /* GIVEN_NAME */);
            }
            if (props.requiredAttributes.lastUpdateTime) {
                stdAttributes.push("updated_at" /* LAST_UPDATE_TIME */);
            }
            if (props.requiredAttributes.locale) {
                stdAttributes.push("locale" /* LOCALE */);
            }
            if (props.requiredAttributes.middleName) {
                stdAttributes.push("middle_name" /* MIDDLE_NAME */);
            }
            if (props.requiredAttributes.nickname) {
                stdAttributes.push("nickname" /* NICKNAME */);
            }
            if (props.requiredAttributes.phoneNumber) {
                stdAttributes.push("phone_number" /* PHONE_NUMBER */);
            }
            if (props.requiredAttributes.preferredUsername) {
                stdAttributes.push("preferred_username" /* PREFERRED_USERNAME */);
            }
            if (props.requiredAttributes.profilePage) {
                stdAttributes.push("profile" /* PROFILE_URL */);
            }
            if (props.requiredAttributes.profilePicture) {
                stdAttributes.push("picture" /* PICTURE_URL */);
            }
            if (props.requiredAttributes.timezone) {
                stdAttributes.push("zoneinfo" /* TIMEZONE */);
            }
            if (props.requiredAttributes.website) {
                stdAttributes.push("website" /* WEBSITE */);
            }
            schema.push(...stdAttributes.map((attr) => {
                return { name: attr, required: true };
            }));
        }
        if (props.customAttributes) {
            const customAttrs = Object.keys(props.customAttributes).map((attrName) => {
                var _a, _b, _c, _d, _e, _f, _g, _h;
                const attrConfig = props.customAttributes[attrName].bind();
                const numberConstraints = {
                    minValue: (_b = (_a = attrConfig.numberConstraints) === null || _a === void 0 ? void 0 : _a.min) === null || _b === void 0 ? void 0 : _b.toString(),
                    maxValue: (_d = (_c = attrConfig.numberConstraints) === null || _c === void 0 ? void 0 : _c.max) === null || _d === void 0 ? void 0 : _d.toString(),
                };
                const stringConstraints = {
                    minLength: (_f = (_e = attrConfig.stringConstraints) === null || _e === void 0 ? void 0 : _e.minLen) === null || _f === void 0 ? void 0 : _f.toString(),
                    maxLength: (_h = (_g = attrConfig.stringConstraints) === null || _g === void 0 ? void 0 : _g.maxLen) === null || _h === void 0 ? void 0 : _h.toString(),
                };
                return {
                    name: attrName,
                    attributeDataType: attrConfig.dataType,
                    numberAttributeConstraints: (attrConfig.numberConstraints) ? numberConstraints : undefined,
                    stringAttributeConstraints: (attrConfig.stringConstraints) ? stringConstraints : undefined,
                    mutable: attrConfig.mutable,
                };
            });
            schema.push(...customAttrs);
        }
        if (schema.length === 0) {
            return undefined;
        }
        return schema;
    }
}
exports.UserPool = UserPool;
function undefinedIfNoKeys(struct) {
    const allUndefined = Object.values(struct).reduce((acc, v) => acc && (v === undefined), true);
    return allUndefined ? undefined : struct;
}
//# sourceMappingURL=data:application/json;base64,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