"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        const actionScope = new cdk.Construct(this.scope, action.actionProperties.actionName);
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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