"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const case_1 = require("case");
const action_1 = require("../action");
const common_1 = require("../common");
// Class copied verbatim from the aws-s3-deployment module.
// Yes, it sucks that we didn't abstract this properly in a common class,
// but having 2 different CacheControl classes that behave differently would be worse I think.
// Something to do when CDK 2.0.0 comes out.
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * Use the provided static factory methods to construct instances of this class.
 * Used in the {@link S3DeployActionProps.cacheControl} property.
 *
 * @see https://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.9
 */
class CacheControl {
    /** @param value the actual text value of the created directive */
    constructor(value) {
        this.value = value;
    }
    /** The 'must-revalidate' cache control directive. */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /** The 'no-cache' cache control directive. */
    static noCache() { return new CacheControl('no-cache'); }
    /** The 'no-transform' cache control directive. */
    static noTransform() { return new CacheControl('no-transform'); }
    /** The 'public' cache control directive. */
    static setPublic() { return new CacheControl('public'); }
    /** The 'private' cache control directive. */
    static setPrivate() { return new CacheControl('private'); }
    /** The 'proxy-revalidate' cache control directive. */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /** The 'max-age' cache control directive. */
    static maxAge(t) { return new CacheControl(`max-age: ${t.toSeconds()}`); }
    /** The 's-max-age' cache control directive. */
    static sMaxAge(t) { return new CacheControl(`s-max-age: ${t.toSeconds()}`); }
    /**
     * Allows you to create an arbitrary cache control directive,
     * in case our support is missing a method for a particular directive.
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Deploys the sourceArtifact to Amazon S3.
 */
class S3DeployAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.DEPLOY,
            provider: 'S3',
            artifactBounds: common_1.deployArtifactBounds(),
            inputs: [props.input],
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        // pipeline needs permissions to write to the S3 bucket
        this.props.bucket.grantWrite(options.role);
        // the Action Role also needs to read from the Pipeline's bucket
        options.bucket.grantRead(options.role);
        const acl = this.props.accessControl;
        return {
            configuration: {
                BucketName: this.props.bucket.bucketName,
                Extract: this.props.extract === false ? 'false' : 'true',
                ObjectKey: this.props.objectKey,
                CannedACL: acl ? case_1.kebab(acl.toString()) : undefined,
                CacheControl: this.props.cacheControl && this.props.cacheControl.map(ac => ac.value).join(', '),
            },
        };
    }
}
exports.S3DeployAction = S3DeployAction;
//# sourceMappingURL=data:application/json;base64,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