"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const custom_action_registration_1 = require("../custom-action-registration");
class BaseJenkinsProvider extends cdk.Construct {
    constructor(scope, id, version) {
        super(scope, id);
        this.version = version || '1';
    }
}
exports.BaseJenkinsProvider = BaseJenkinsProvider;
/**
 * A class representing Jenkins providers.
 *
 * @see #import
 */
class JenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.buildIncluded = false;
        this.testIncluded = false;
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
        if (props.forBuild === true) {
            this._registerBuildProvider();
        }
        if (props.forTest === true) {
            this._registerTestProvider();
        }
    }
    /**
     * Import a Jenkins provider registered either outside the CDK,
     * or in a different CDK Stack.
     *
     * @param scope the parent Construct for the new provider
     * @param id the identifier of the new provider Construct
     * @param attrs the properties used to identify the existing provider
     * @returns a new Construct representing a reference to an existing Jenkins provider
     */
    static fromJenkinsProviderAttributes(scope, id, attrs) {
        return new ImportedJenkinsProvider(scope, id, attrs);
    }
    /**
     * @internal
     */
    _registerBuildProvider() {
        if (this.buildIncluded) {
            return;
        }
        this.buildIncluded = true;
        this.registerJenkinsCustomAction('JenkinsBuildProviderResource', codepipeline.ActionCategory.BUILD);
    }
    /**
     * @internal
     */
    _registerTestProvider() {
        if (this.testIncluded) {
            return;
        }
        this.testIncluded = true;
        this.registerJenkinsCustomAction('JenkinsTestProviderResource', codepipeline.ActionCategory.TEST);
    }
    registerJenkinsCustomAction(id, category) {
        new custom_action_registration_1.CustomActionRegistration(this, id, {
            category,
            artifactBounds: exports.jenkinsArtifactsBounds,
            provider: this.providerName,
            version: this.version,
            entityUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}'),
            executionUrl: appendToUrl(this.serverUrl, 'job/{Config:ProjectName}/{ExternalExecutionId}'),
            actionProperties: [
                {
                    name: 'ProjectName',
                    required: true,
                    key: true,
                    queryable: true,
                },
            ],
        });
    }
}
exports.JenkinsProvider = JenkinsProvider;
class ImportedJenkinsProvider extends BaseJenkinsProvider {
    constructor(scope, id, props) {
        super(scope, id, props.version);
        this.providerName = props.providerName;
        this.serverUrl = props.serverUrl;
    }
    _registerBuildProvider() {
        // do nothing
    }
    _registerTestProvider() {
        // do nothing
    }
}
function appendToUrl(baseUrl, path) {
    return baseUrl.endsWith('/') ? baseUrl + path : `${baseUrl}/${path}`;
}
exports.jenkinsArtifactsBounds = {
    minInputs: 0,
    maxInputs: 5,
    minOutputs: 0,
    maxOutputs: 5,
};
//# sourceMappingURL=data:application/json;base64,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