"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * If and how the GitHub source action should be triggered
 */
var GitHubTrigger;
(function (GitHubTrigger) {
    GitHubTrigger["NONE"] = "None";
    GitHubTrigger["POLL"] = "Poll";
    GitHubTrigger["WEBHOOK"] = "WebHook";
})(GitHubTrigger = exports.GitHubTrigger || (exports.GitHubTrigger = {}));
/**
 * Source that is provided by a GitHub repository.
 */
class GitHubSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            category: codepipeline.ActionCategory.SOURCE,
            owner: 'ThirdParty',
            provider: 'GitHub',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
            commitUrl: this.variableExpression('CommitUrl'),
        };
    }
    bound(scope, stage, _options) {
        if (!this.props.trigger || this.props.trigger === GitHubTrigger.WEBHOOK) {
            new codepipeline.CfnWebhook(scope, 'WebhookResource', {
                authentication: 'GITHUB_HMAC',
                authenticationConfiguration: {
                    secretToken: this.props.oauthToken.toString(),
                },
                filters: [
                    {
                        jsonPath: '$.ref',
                        matchEquals: 'refs/heads/{Branch}',
                    },
                ],
                targetAction: this.actionProperties.actionName,
                targetPipeline: stage.pipeline.pipelineName,
                targetPipelineVersion: 1,
                registerWithThirdParty: true,
            });
        }
        return {
            configuration: {
                Owner: this.props.owner,
                Repo: this.props.repo,
                Branch: this.props.branch || 'master',
                OAuthToken: this.props.oauthToken.toString(),
                PollForSourceChanges: this.props.trigger === GitHubTrigger.POLL,
            },
        };
    }
}
exports.GitHubSourceAction = GitHubSourceAction;
//# sourceMappingURL=data:application/json;base64,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