"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * The generations of AWS load balancing solutions.
 */
var LoadBalancerGeneration;
(function (LoadBalancerGeneration) {
    /**
     * The first generation (ELB Classic).
     */
    LoadBalancerGeneration[LoadBalancerGeneration["FIRST"] = 0] = "FIRST";
    /**
     * The second generation (ALB and NLB).
     */
    LoadBalancerGeneration[LoadBalancerGeneration["SECOND"] = 1] = "SECOND";
})(LoadBalancerGeneration = exports.LoadBalancerGeneration || (exports.LoadBalancerGeneration = {}));
/**
 * An interface of an abstract load balancer, as needed by CodeDeploy.
 * Create instances using the static factory methods:
 * {@link #classic}, {@link #application} and {@link #network}.
 */
class LoadBalancer {
    /**
     * Creates a new CodeDeploy load balancer from a Classic ELB Load Balancer.
     *
     * @param loadBalancer a classic ELB Load Balancer
     */
    static classic(loadBalancer) {
        class ClassicLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.FIRST;
                this.name = loadBalancer.loadBalancerName;
            }
        }
        return new ClassicLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from an Application Load Balancer Target Group.
     *
     * @param albTargetGroup an ALB Target Group
     */
    static application(albTargetGroup) {
        class AlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = albTargetGroup.targetGroupName;
            }
        }
        return new AlbLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from a Network Load Balancer Target Group.
     *
     * @param nlbTargetGroup an NLB Target Group
     */
    static network(nlbTargetGroup) {
        class NlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = nlbTargetGroup.targetGroupName;
            }
        }
        return new NlbLoadBalancer();
    }
}
exports.LoadBalancer = LoadBalancer;
//# sourceMappingURL=data:application/json;base64,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