"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const codebuild_generated_1 = require("./codebuild.generated");
const report_group_utils_1 = require("./report-group-utils");
class ReportGroupBase extends cdk.Resource {
    grantWrite(identity) {
        const ret = iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'codebuild:CreateReport',
                'codebuild:UpdateReport',
                'codebuild:BatchPutTestCases',
            ],
            resourceArns: [this.reportGroupArn],
        });
        if (this.exportBucket) {
            this.exportBucket.grantWrite(identity);
        }
        return ret;
    }
}
/**
 * The ReportGroup resource class.
 */
class ReportGroup extends ReportGroupBase {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: props.reportGroupName,
        });
        const resource = new codebuild_generated_1.CfnReportGroup(this, 'Resource', {
            type: 'TEST',
            exportConfig: {
                exportConfigType: props.exportBucket ? 'S3' : 'NO_EXPORT',
                s3Destination: props.exportBucket
                    ? {
                        bucket: props.exportBucket.bucketName,
                        encryptionDisabled: props.exportBucket.encryptionKey ? false : undefined,
                        encryptionKey: (_a = props.exportBucket.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
                        packaging: props.zipExport ? 'ZIP' : undefined,
                    }
                    : undefined,
            },
        });
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: cdk.RemovalPolicy.RETAIN,
        });
        this.reportGroupArn = this.getResourceArnAttribute(resource.attrArn, report_group_utils_1.reportGroupArnComponents(this.physicalName));
        this.reportGroupName = this.getResourceNameAttribute(
        // there is no separate name attribute,
        // so use Fn::Select + Fn::Split to make one
        cdk.Fn.select(1, cdk.Fn.split('/', resource.ref)));
        this.exportBucket = props.exportBucket;
    }
    /**
     * Reference an existing ReportGroup,
     * defined outside of the CDK code,
     * by name.
     */
    static fromReportGroupName(scope, id, reportGroupName) {
        class Import extends ReportGroupBase {
            constructor() {
                super(...arguments);
                this.reportGroupName = reportGroupName;
                this.reportGroupArn = report_group_utils_1.renderReportGroupArn(scope, reportGroupName);
                this.exportBucket = undefined;
            }
        }
        return new Import(scope, id);
    }
}
exports.ReportGroup = ReportGroup;
//# sourceMappingURL=data:application/json;base64,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