"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const metric_types_1 = require("../metric-types");
/**
 * Parse a statistic, returning the type of metric that was used (simple or percentile)
 */
function parseStatistic(stat) {
    const lowerStat = stat.toLowerCase();
    // Simple statistics
    const statMap = {
        average: metric_types_1.Statistic.AVERAGE,
        avg: metric_types_1.Statistic.AVERAGE,
        minimum: metric_types_1.Statistic.MINIMUM,
        min: metric_types_1.Statistic.MINIMUM,
        maximum: metric_types_1.Statistic.MAXIMUM,
        max: metric_types_1.Statistic.MAXIMUM,
        samplecount: metric_types_1.Statistic.SAMPLE_COUNT,
        n: metric_types_1.Statistic.SAMPLE_COUNT,
        sum: metric_types_1.Statistic.SUM,
    };
    if (lowerStat in statMap) {
        return {
            type: 'simple',
            statistic: statMap[lowerStat],
        };
    }
    // Percentile statistics
    const re = /^p([\d.]+)$/;
    const m = re.exec(lowerStat);
    if (m) {
        return {
            type: 'percentile',
            percentile: parseFloat(m[1]),
        };
    }
    throw new Error(`Not a valid statistic: '${stat}', must be one of Average | Minimum | Maximum | SampleCount | Sum | pNN.NN`);
}
exports.parseStatistic = parseStatistic;
function normalizeStatistic(stat) {
    const parsed = parseStatistic(stat);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    else {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return stat.toLowerCase();
    }
}
exports.normalizeStatistic = normalizeStatistic;
//# sourceMappingURL=data:application/json;base64,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