"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const widget_1 = require("./widget");
// This file contains widgets that exist for layout purposes
/**
 * A widget that contains other widgets in a horizontal row
 *
 * Widgets will be laid out next to each other
 */
class Row {
    constructor(...widgets) {
        /**
         * Relative position of each widget inside this row
         */
        this.offsets = [];
        this.widgets = widgets;
        this.width = 0;
        this.height = 0;
        let x = 0;
        let y = 0;
        for (const widget of widgets) {
            // See if we need to horizontally wrap to add this widget
            if (x + widget.width > widget_1.GRID_WIDTH) {
                y = this.height;
                x = 0;
            }
            this.offsets.push({ x, y });
            this.width = Math.max(this.width, x + widget.width);
            this.height = Math.max(this.height, y + widget.height);
            x += widget.width;
        }
    }
    position(x, y) {
        for (let i = 0; i < this.widgets.length; i++) {
            this.widgets[i].position(x + this.offsets[i].x, y + this.offsets[i].y);
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Row = Row;
/**
 * A widget that contains other widgets in a vertical column
 *
 * Widgets will be laid out next to each other
 */
class Column {
    constructor(...widgets) {
        this.widgets = widgets;
        // There's no vertical wrapping so this one's a lot easier
        this.width = Math.max(...this.widgets.map(w => w.width));
        this.height = sum(...this.widgets.map(w => w.height));
    }
    position(x, y) {
        let widgetY = y;
        for (const widget of this.widgets) {
            widget.position(x, widgetY);
            widgetY += widget.height;
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Column = Column;
/**
 * A widget that doesn't display anything but takes up space
 */
class Spacer {
    constructor(props = {}) {
        this.width = props.width || 1;
        this.height = props.height || 1;
    }
    position(_x, _y) {
        // Don't need to do anything, not a physical widget
    }
    toJson() {
        return [];
    }
}
exports.Spacer = Spacer;
/**
 * Return the sum of a list of numbers
 */
function sum(...xs) {
    let ret = 0;
    for (const x of xs) {
        ret += x;
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibGF5b3V0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibGF5b3V0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEscUNBQStDO0FBQy9DLDREQUE0RDtBQUM1RDs7OztHQUlHO0FBQ0gsTUFBYSxHQUFHO0lBV1osWUFBWSxHQUFHLE9BQWtCO1FBSmpDOztXQUVHO1FBQ2MsWUFBTyxHQUFhLEVBQUUsQ0FBQztRQUVwQyxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztRQUN2QixJQUFJLENBQUMsS0FBSyxHQUFHLENBQUMsQ0FBQztRQUNmLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO1FBQ2hCLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNWLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNWLEtBQUssTUFBTSxNQUFNLElBQUksT0FBTyxFQUFFO1lBQzFCLHlEQUF5RDtZQUN6RCxJQUFJLENBQUMsR0FBRyxNQUFNLENBQUMsS0FBSyxHQUFHLG1CQUFVLEVBQUU7Z0JBQy9CLENBQUMsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDO2dCQUNoQixDQUFDLEdBQUcsQ0FBQyxDQUFDO2FBQ1Q7WUFDRCxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQzVCLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLENBQUMsR0FBRyxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDcEQsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUN2RCxDQUFDLElBQUksTUFBTSxDQUFDLEtBQUssQ0FBQztTQUNyQjtJQUNMLENBQUM7SUFDTSxRQUFRLENBQUMsQ0FBUyxFQUFFLENBQVM7UUFDaEMsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFO1lBQzFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUMxRTtJQUNMLENBQUM7SUFDTSxNQUFNO1FBQ1QsTUFBTSxHQUFHLEdBQVUsRUFBRSxDQUFDO1FBQ3RCLEtBQUssTUFBTSxNQUFNLElBQUksSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUMvQixHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsTUFBTSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7U0FDaEM7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7Q0FDSjtBQXpDRCxrQkF5Q0M7QUFDRDs7OztHQUlHO0FBQ0gsTUFBYSxNQUFNO0lBT2YsWUFBWSxHQUFHLE9BQWtCO1FBQzdCLElBQUksQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDO1FBQ3ZCLDBEQUEwRDtRQUMxRCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ3pELElBQUksQ0FBQyxNQUFNLEdBQUcsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztJQUMxRCxDQUFDO0lBQ00sUUFBUSxDQUFDLENBQVMsRUFBRSxDQUFTO1FBQ2hDLElBQUksT0FBTyxHQUFHLENBQUMsQ0FBQztRQUNoQixLQUFLLE1BQU0sTUFBTSxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDL0IsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDNUIsT0FBTyxJQUFJLE1BQU0sQ0FBQyxNQUFNLENBQUM7U0FDNUI7SUFDTCxDQUFDO0lBQ00sTUFBTTtRQUNULE1BQU0sR0FBRyxHQUFVLEVBQUUsQ0FBQztRQUN0QixLQUFLLE1BQU0sTUFBTSxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDL0IsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1NBQ2hDO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0NBQ0o7QUEzQkQsd0JBMkJDO0FBa0JEOztHQUVHO0FBQ0gsTUFBYSxNQUFNO0lBR2YsWUFBWSxRQUFxQixFQUFFO1FBQy9CLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssSUFBSSxDQUFDLENBQUM7UUFDOUIsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxJQUFJLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBQ00sUUFBUSxDQUFDLEVBQVUsRUFBRSxFQUFVO1FBQ2xDLG1EQUFtRDtJQUN2RCxDQUFDO0lBQ00sTUFBTTtRQUNULE9BQU8sRUFBRSxDQUFDO0lBQ2QsQ0FBQztDQUNKO0FBYkQsd0JBYUM7QUFRRDs7R0FFRztBQUNILFNBQVMsR0FBRyxDQUFDLEdBQUcsRUFBWTtJQUN4QixJQUFJLEdBQUcsR0FBRyxDQUFDLENBQUM7SUFDWixLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsRUFBRTtRQUNoQixHQUFHLElBQUksQ0FBQyxDQUFDO0tBQ1o7SUFDRCxPQUFPLEdBQUcsQ0FBQztBQUNmLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBHUklEX1dJRFRILCBJV2lkZ2V0IH0gZnJvbSAnLi93aWRnZXQnO1xuLy8gVGhpcyBmaWxlIGNvbnRhaW5zIHdpZGdldHMgdGhhdCBleGlzdCBmb3IgbGF5b3V0IHB1cnBvc2VzXG4vKipcbiAqIEEgd2lkZ2V0IHRoYXQgY29udGFpbnMgb3RoZXIgd2lkZ2V0cyBpbiBhIGhvcml6b250YWwgcm93XG4gKlxuICogV2lkZ2V0cyB3aWxsIGJlIGxhaWQgb3V0IG5leHQgdG8gZWFjaCBvdGhlclxuICovXG5leHBvcnQgY2xhc3MgUm93IGltcGxlbWVudHMgSVdpZGdldCB7XG4gICAgcHVibGljIHJlYWRvbmx5IHdpZHRoOiBudW1iZXI7XG4gICAgcHVibGljIHJlYWRvbmx5IGhlaWdodDogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIExpc3Qgb2YgY29udGFpbmVkIHdpZGdldHNcbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IHdpZGdldHM6IElXaWRnZXRbXTtcbiAgICAvKipcbiAgICAgKiBSZWxhdGl2ZSBwb3NpdGlvbiBvZiBlYWNoIHdpZGdldCBpbnNpZGUgdGhpcyByb3dcbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IG9mZnNldHM6IFZlY3RvcltdID0gW107XG4gICAgY29uc3RydWN0b3IoLi4ud2lkZ2V0czogSVdpZGdldFtdKSB7XG4gICAgICAgIHRoaXMud2lkZ2V0cyA9IHdpZGdldHM7XG4gICAgICAgIHRoaXMud2lkdGggPSAwO1xuICAgICAgICB0aGlzLmhlaWdodCA9IDA7XG4gICAgICAgIGxldCB4ID0gMDtcbiAgICAgICAgbGV0IHkgPSAwO1xuICAgICAgICBmb3IgKGNvbnN0IHdpZGdldCBvZiB3aWRnZXRzKSB7XG4gICAgICAgICAgICAvLyBTZWUgaWYgd2UgbmVlZCB0byBob3Jpem9udGFsbHkgd3JhcCB0byBhZGQgdGhpcyB3aWRnZXRcbiAgICAgICAgICAgIGlmICh4ICsgd2lkZ2V0LndpZHRoID4gR1JJRF9XSURUSCkge1xuICAgICAgICAgICAgICAgIHkgPSB0aGlzLmhlaWdodDtcbiAgICAgICAgICAgICAgICB4ID0gMDtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIHRoaXMub2Zmc2V0cy5wdXNoKHsgeCwgeSB9KTtcbiAgICAgICAgICAgIHRoaXMud2lkdGggPSBNYXRoLm1heCh0aGlzLndpZHRoLCB4ICsgd2lkZ2V0LndpZHRoKTtcbiAgICAgICAgICAgIHRoaXMuaGVpZ2h0ID0gTWF0aC5tYXgodGhpcy5oZWlnaHQsIHkgKyB3aWRnZXQuaGVpZ2h0KTtcbiAgICAgICAgICAgIHggKz0gd2lkZ2V0LndpZHRoO1xuICAgICAgICB9XG4gICAgfVxuICAgIHB1YmxpYyBwb3NpdGlvbih4OiBudW1iZXIsIHk6IG51bWJlcik6IHZvaWQge1xuICAgICAgICBmb3IgKGxldCBpID0gMDsgaSA8IHRoaXMud2lkZ2V0cy5sZW5ndGg7IGkrKykge1xuICAgICAgICAgICAgdGhpcy53aWRnZXRzW2ldLnBvc2l0aW9uKHggKyB0aGlzLm9mZnNldHNbaV0ueCwgeSArIHRoaXMub2Zmc2V0c1tpXS55KTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwdWJsaWMgdG9Kc29uKCk6IGFueVtdIHtcbiAgICAgICAgY29uc3QgcmV0OiBhbnlbXSA9IFtdO1xuICAgICAgICBmb3IgKGNvbnN0IHdpZGdldCBvZiB0aGlzLndpZGdldHMpIHtcbiAgICAgICAgICAgIHJldC5wdXNoKC4uLndpZGdldC50b0pzb24oKSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG59XG4vKipcbiAqIEEgd2lkZ2V0IHRoYXQgY29udGFpbnMgb3RoZXIgd2lkZ2V0cyBpbiBhIHZlcnRpY2FsIGNvbHVtblxuICpcbiAqIFdpZGdldHMgd2lsbCBiZSBsYWlkIG91dCBuZXh0IHRvIGVhY2ggb3RoZXJcbiAqL1xuZXhwb3J0IGNsYXNzIENvbHVtbiBpbXBsZW1lbnRzIElXaWRnZXQge1xuICAgIHB1YmxpYyByZWFkb25seSB3aWR0aDogbnVtYmVyO1xuICAgIHB1YmxpYyByZWFkb25seSBoZWlnaHQ6IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBMaXN0IG9mIGNvbnRhaW5lZCB3aWRnZXRzXG4gICAgICovXG4gICAgcHJpdmF0ZSByZWFkb25seSB3aWRnZXRzOiBJV2lkZ2V0W107XG4gICAgY29uc3RydWN0b3IoLi4ud2lkZ2V0czogSVdpZGdldFtdKSB7XG4gICAgICAgIHRoaXMud2lkZ2V0cyA9IHdpZGdldHM7XG4gICAgICAgIC8vIFRoZXJlJ3Mgbm8gdmVydGljYWwgd3JhcHBpbmcgc28gdGhpcyBvbmUncyBhIGxvdCBlYXNpZXJcbiAgICAgICAgdGhpcy53aWR0aCA9IE1hdGgubWF4KC4uLnRoaXMud2lkZ2V0cy5tYXAodyA9PiB3LndpZHRoKSk7XG4gICAgICAgIHRoaXMuaGVpZ2h0ID0gc3VtKC4uLnRoaXMud2lkZ2V0cy5tYXAodyA9PiB3LmhlaWdodCkpO1xuICAgIH1cbiAgICBwdWJsaWMgcG9zaXRpb24oeDogbnVtYmVyLCB5OiBudW1iZXIpOiB2b2lkIHtcbiAgICAgICAgbGV0IHdpZGdldFkgPSB5O1xuICAgICAgICBmb3IgKGNvbnN0IHdpZGdldCBvZiB0aGlzLndpZGdldHMpIHtcbiAgICAgICAgICAgIHdpZGdldC5wb3NpdGlvbih4LCB3aWRnZXRZKTtcbiAgICAgICAgICAgIHdpZGdldFkgKz0gd2lkZ2V0LmhlaWdodDtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwdWJsaWMgdG9Kc29uKCk6IGFueVtdIHtcbiAgICAgICAgY29uc3QgcmV0OiBhbnlbXSA9IFtdO1xuICAgICAgICBmb3IgKGNvbnN0IHdpZGdldCBvZiB0aGlzLndpZGdldHMpIHtcbiAgICAgICAgICAgIHJldC5wdXNoKC4uLndpZGdldC50b0pzb24oKSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG59XG4vKipcbiAqIFByb3BzIG9mIHRoZSBzcGFjZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTcGFjZXJQcm9wcyB7XG4gICAgLyoqXG4gICAgICogV2lkdGggb2YgdGhlIHNwYWNlclxuICAgICAqXG4gICAgICogQGRlZmF1bHQgMVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHdpZHRoPzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIEhlaWdodCBvZiB0aGUgc3BhY2VyXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdDogMVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGhlaWdodD86IG51bWJlcjtcbn1cbi8qKlxuICogQSB3aWRnZXQgdGhhdCBkb2Vzbid0IGRpc3BsYXkgYW55dGhpbmcgYnV0IHRha2VzIHVwIHNwYWNlXG4gKi9cbmV4cG9ydCBjbGFzcyBTcGFjZXIgaW1wbGVtZW50cyBJV2lkZ2V0IHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgd2lkdGg6IG51bWJlcjtcbiAgICBwdWJsaWMgcmVhZG9ubHkgaGVpZ2h0OiBudW1iZXI7XG4gICAgY29uc3RydWN0b3IocHJvcHM6IFNwYWNlclByb3BzID0ge30pIHtcbiAgICAgICAgdGhpcy53aWR0aCA9IHByb3BzLndpZHRoIHx8IDE7XG4gICAgICAgIHRoaXMuaGVpZ2h0ID0gcHJvcHMuaGVpZ2h0IHx8IDE7XG4gICAgfVxuICAgIHB1YmxpYyBwb3NpdGlvbihfeDogbnVtYmVyLCBfeTogbnVtYmVyKTogdm9pZCB7XG4gICAgICAgIC8vIERvbid0IG5lZWQgdG8gZG8gYW55dGhpbmcsIG5vdCBhIHBoeXNpY2FsIHdpZGdldFxuICAgIH1cbiAgICBwdWJsaWMgdG9Kc29uKCk6IGFueVtdIHtcbiAgICAgICAgcmV0dXJuIFtdO1xuICAgIH1cbn1cbi8qKlxuICogSW50ZXJmYWNlIHJlcHJlc2VudGluZyBhIDJEIHZlY3RvciAoZm9yIGludGVybmFsIHVzZSlcbiAqL1xuaW50ZXJmYWNlIFZlY3RvciB7XG4gICAgeDogbnVtYmVyO1xuICAgIHk6IG51bWJlcjtcbn1cbi8qKlxuICogUmV0dXJuIHRoZSBzdW0gb2YgYSBsaXN0IG9mIG51bWJlcnNcbiAqL1xuZnVuY3Rpb24gc3VtKC4uLnhzOiBudW1iZXJbXSkge1xuICAgIGxldCByZXQgPSAwO1xuICAgIGZvciAoY29uc3QgeCBvZiB4cykge1xuICAgICAgICByZXQgKz0geDtcbiAgICB9XG4gICAgcmV0dXJuIHJldDtcbn1cbiJdfQ==