"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * Comparison operator for evaluating alarms
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    /**
     * Specified statistic is greater than or equal to the threshold
     */
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    /**
     * Specified statistic is strictly greater than the threshold
     */
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    /**
     * Specified statistic is strictly less than the threshold
     */
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    /**
     * Specified statistic is less than or equal to the threshold.
     */
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
    /**
     * Specified statistic is lower than or greater than the anomaly model band.
     * Used only for alarms based on anomaly detection models
     */
    ComparisonOperator["LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD"] = "LessThanLowerOrGreaterThanUpperThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '>=',
};
/**
 * Specify how missing data points are treated during alarm evaluation
 */
var TreatMissingData;
(function (TreatMissingData) {
    /**
     * Missing data points are treated as breaching the threshold
     */
    TreatMissingData["BREACHING"] = "breaching";
    /**
     * Missing data points are treated as being within the threshold
     */
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    /**
     * The current alarm state is maintained
     */
    TreatMissingData["IGNORE"] = "ignore";
    /**
     * The alarm does not consider missing data points when evaluating whether to change state
     */
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * An alarm on a CloudWatch metric
 */
class Alarm extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        // Render metric, process potential overrides from the alarm
        // (It would be preferable if the statistic etc. was worked into the metric,
        // but hey we're allowing overrides...)
        const metricProps = this.renderMetric(props.metric);
        if (props.period) {
            metricProps.period = props.period.toSeconds();
        }
        if (props.statistic) {
            // Will overwrite both fields if present
            Object.assign(metricProps, {
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            });
        }
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.listValue({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.listValue({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.listValue({ produce: () => this.okActionArns }),
            // Metric
            ...metricProps,
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        const datapoints = props.datapointsToAlarm || props.evaluationPeriods;
        this.annotation = {
            // tslint:disable-next-line:max-line-length
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${datapoints} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    /**
     * Import an existing CloudWatch alarm provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name
     * @param alarmArn Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm:Foo)
     */
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(alarmArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Trigger this action if the alarm fires
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addAlarmAction(...actions) {
        if (this.alarmActionArns === undefined) {
            this.alarmActionArns = [];
        }
        this.alarmActionArns.push(...actions.map(a => a.bind(this, this).alarmActionArn));
    }
    /**
     * Trigger this action if there is insufficient data to evaluate the alarm
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addInsufficientDataAction(...actions) {
        if (this.insufficientDataActionArns === undefined) {
            this.insufficientDataActionArns = [];
        }
        this.insufficientDataActionArns.push(...actions.map(a => a.bind(this, this).alarmActionArn));
    }
    /**
     * Trigger this action if the alarm returns from breaching state into ok state
     *
     * Typically the ARN of an SNS topic or ARN of an AutoScaling policy.
     */
    addOkAction(...actions) {
        if (this.okActionArns === undefined) {
            this.okActionArns = [];
        }
        this.okActionArns.push(...actions.map(a => a.bind(this, this).alarmActionArn));
    }
    /**
     * Turn this alarm into a horizontal annotation
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *   you have both a "small margin/long period" alarm as well as a
     *   "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     */
    toAnnotation() {
        return this.annotation;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(st) {
                var _a;
                self.validateMetricStat(st, metric);
                return object_1.dropUndefined({
                    dimensions: st.dimensions,
                    namespace: st.namespace,
                    metricName: st.metricName,
                    period: (_a = st.period) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                    statistic: renderIfSimpleStatistic(st.statistic),
                    extendedStatistic: renderIfExtendedStatistic(st.statistic),
                    unit: st.unitFilter,
                });
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _a;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _a;
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                period: mathExprHasSubmetrics(expr) ? undefined : expr.period,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    })),
                };
            },
        });
    }
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
        if (definitelyDifferent(stat.account, stack.account)) {
            throw new Error(`Cannot create an Alarm in account '${stack.account}' based on metric '${metric}' in '${stat.account}'`);
        }
    }
}
exports.Alarm = Alarm;
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    return undefined;
}
function mathExprHasSubmetrics(expr) {
    return Object.keys(expr.usingMetrics).length > 0;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxhcm0uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbGFybS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHFDQUFnRixDQUFDLGdEQUFnRDtBQUVqSSxpRUFBaUU7QUFJakUsdURBQXFFO0FBQ3JFLDZDQUFpRDtBQUNqRCxtREFBZ0Q7QUFDaEQsbURBQXFEO0FBOEJyRDs7R0FFRztBQUNILElBQVksa0JBc0JYO0FBdEJELFdBQVksa0JBQWtCO0lBQzFCOztPQUVHO0lBQ0gsMEZBQW9FLENBQUE7SUFDcEU7O09BRUc7SUFDSCxxRUFBK0MsQ0FBQTtJQUMvQzs7T0FFRztJQUNILCtEQUF5QyxDQUFBO0lBQ3pDOztPQUVHO0lBQ0gsb0ZBQThELENBQUE7SUFDOUQ7OztPQUdHO0lBQ0gsa0hBQTRGLENBQUE7QUFDaEcsQ0FBQyxFQXRCVyxrQkFBa0IsR0FBbEIsMEJBQWtCLEtBQWxCLDBCQUFrQixRQXNCN0I7QUFDRCxNQUFNLGdCQUFnQixHQUVsQjtJQUNBLDZCQUE2QixFQUFFLElBQUk7SUFDbkMsb0JBQW9CLEVBQUUsR0FBRztJQUN6QixpQkFBaUIsRUFBRSxHQUFHO0lBQ3RCLDBCQUEwQixFQUFFLElBQUk7Q0FDbkMsQ0FBQztBQUNGOztHQUVHO0FBQ0gsSUFBWSxnQkFpQlg7QUFqQkQsV0FBWSxnQkFBZ0I7SUFDeEI7O09BRUc7SUFDSCwyQ0FBdUIsQ0FBQTtJQUN2Qjs7T0FFRztJQUNILGtEQUE4QixDQUFBO0lBQzlCOztPQUVHO0lBQ0gscUNBQWlCLENBQUE7SUFDakI7O09BRUc7SUFDSCx1Q0FBbUIsQ0FBQTtBQUN2QixDQUFDLEVBakJXLGdCQUFnQixHQUFoQix3QkFBZ0IsS0FBaEIsd0JBQWdCLFFBaUIzQjtBQUNEOztHQUVHO0FBQ0gsTUFBYSxLQUFNLFNBQVEsZUFBUTtJQXNDL0IsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFpQjtRQUN2RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNiLFlBQVksRUFBRSxLQUFLLENBQUMsU0FBUztTQUNoQyxDQUFDLENBQUM7UUFDSCxNQUFNLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxrQkFBa0IsQ0FBQyxrQ0FBa0MsQ0FBQztRQUM3Ryw0REFBNEQ7UUFDNUQsNEVBQTRFO1FBQzVFLHVDQUF1QztRQUN2QyxNQUFNLFdBQVcsR0FBc0MsSUFBSSxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDdkYsSUFBSSxLQUFLLENBQUMsTUFBTSxFQUFFO1lBQ2QsV0FBVyxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxDQUFDO1NBQ2pEO1FBQ0QsSUFBSSxLQUFLLENBQUMsU0FBUyxFQUFFO1lBQ2pCLHdDQUF3QztZQUN4QyxNQUFNLENBQUMsTUFBTSxDQUFDLFdBQVcsRUFBRTtnQkFDdkIsU0FBUyxFQUFFLHVCQUF1QixDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7Z0JBQ25ELGlCQUFpQixFQUFFLHlCQUF5QixDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUM7YUFDaEUsQ0FBQyxDQUFDO1NBQ047UUFDRCxNQUFNLEtBQUssR0FBRyxJQUFJLCtCQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN6QyxPQUFPO1lBQ1AsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtZQUN4QyxTQUFTLEVBQUUsSUFBSSxDQUFDLFlBQVk7WUFDNUIsYUFBYTtZQUNiLGtCQUFrQjtZQUNsQixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7WUFDMUIsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtZQUMxQyxnQ0FBZ0MsRUFBRSxLQUFLLENBQUMsZ0NBQWdDO1lBQ3hFLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUI7WUFDMUMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtZQUN4QyxVQUFVO1lBQ1YsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1lBQ3BDLFlBQVksRUFBRSxXQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztZQUNyRSx1QkFBdUIsRUFBRSxXQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLDBCQUEwQixDQUFDLEVBQUUsQ0FBQztZQUM3RixTQUFTLEVBQUUsV0FBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7WUFDL0QsU0FBUztZQUNULEdBQUcsV0FBVztTQUNqQixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFO1lBQ3hELE9BQU8sRUFBRSxZQUFZO1lBQ3JCLFFBQVEsRUFBRSxPQUFPO1lBQ2pCLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUMvQixHQUFHLEVBQUUsR0FBRztTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUMxRCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUM7UUFDM0IsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixJQUFJLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUN0RSxJQUFJLENBQUMsVUFBVSxHQUFHO1lBQ2QsMkNBQTJDO1lBQzNDLEtBQUssRUFBRSxHQUFHLElBQUksQ0FBQyxNQUFNLElBQUksZ0JBQWdCLENBQUMsa0JBQWtCLENBQUMsSUFBSSxLQUFLLENBQUMsU0FBUyxRQUFRLFVBQVUsc0JBQXNCLGNBQWMsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLEdBQUcsMEJBQVksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsU0FBUyxFQUFFLENBQUMsRUFBRTtZQUMxTSxLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7U0FDekIsQ0FBQztJQUNOLENBQUM7SUF6RkQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFnQjtRQUNyRSxNQUFNLE1BQU8sU0FBUSxlQUFRO1lBQTdCOztnQkFDb0IsYUFBUSxHQUFHLFFBQVEsQ0FBQztnQkFDcEIsY0FBUyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxHQUFHLENBQUMsQ0FBQyxZQUFhLENBQUM7WUFDdEYsQ0FBQztTQUFBO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQTZFRDs7OztPQUlHO0lBQ0ksY0FBYyxDQUFDLEdBQUcsT0FBdUI7UUFDNUMsSUFBSSxJQUFJLENBQUMsZUFBZSxLQUFLLFNBQVMsRUFBRTtZQUNwQyxJQUFJLENBQUMsZUFBZSxHQUFHLEVBQUUsQ0FBQztTQUM3QjtRQUNELElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7SUFDdEYsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSx5QkFBeUIsQ0FBQyxHQUFHLE9BQXVCO1FBQ3ZELElBQUksSUFBSSxDQUFDLDBCQUEwQixLQUFLLFNBQVMsRUFBRTtZQUMvQyxJQUFJLENBQUMsMEJBQTBCLEdBQUcsRUFBRSxDQUFDO1NBQ3hDO1FBQ0QsSUFBSSxDQUFDLDBCQUEwQixDQUFDLElBQUksQ0FBQyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDO0lBQ2pHLENBQUM7SUFDRDs7OztPQUlHO0lBQ0ksV0FBVyxDQUFDLEdBQUcsT0FBdUI7UUFDekMsSUFBSSxJQUFJLENBQUMsWUFBWSxLQUFLLFNBQVMsRUFBRTtZQUNqQyxJQUFJLENBQUMsWUFBWSxHQUFHLEVBQUUsQ0FBQztTQUMxQjtRQUNELElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7SUFDbkYsQ0FBQztJQUNEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLFlBQVk7UUFDZixPQUFPLElBQUksQ0FBQyxVQUFVLENBQUM7SUFDM0IsQ0FBQztJQUNPLFlBQVksQ0FBQyxNQUFlO1FBQ2hDLE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUNsQixPQUFPLDRCQUFjLENBQUMsTUFBTSxFQUFFO1lBQzFCLFFBQVEsQ0FBQyxFQUFFOztnQkFDUCxJQUFJLENBQUMsa0JBQWtCLENBQUMsRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUNwQyxPQUFPLHNCQUFhLENBQUM7b0JBQ2pCLFVBQVUsRUFBRSxFQUFFLENBQUMsVUFBVTtvQkFDekIsU0FBUyxFQUFFLEVBQUUsQ0FBQyxTQUFTO29CQUN2QixVQUFVLEVBQUUsRUFBRSxDQUFDLFVBQVU7b0JBQ3pCLE1BQU0sUUFBRSxFQUFFLENBQUMsTUFBTSwwQ0FBRSxTQUFTLEVBQUU7b0JBQzlCLFNBQVMsRUFBRSx1QkFBdUIsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDO29CQUNoRCxpQkFBaUIsRUFBRSx5QkFBeUIsQ0FBQyxFQUFFLENBQUMsU0FBUyxDQUFDO29CQUMxRCxJQUFJLEVBQUUsRUFBRSxDQUFDLFVBQVU7aUJBQ3RCLENBQUMsQ0FBQztZQUNQLENBQUM7WUFDRCxjQUFjO2dCQUNWLCtDQUErQztnQkFDL0MsTUFBTSxJQUFJLEdBQUcsSUFBSSxxQkFBUyxFQUFXLENBQUM7Z0JBQ3RDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUMvQixJQUFJLEdBQUcsR0FBRyxDQUFDLENBQUM7Z0JBQ1osU0FBUyxjQUFjO29CQUNuQixPQUFPLFFBQVEsRUFBRSxHQUFHLEVBQUUsQ0FBQztnQkFDM0IsQ0FBQztnQkFDRCxPQUFPO29CQUNILE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLDRCQUFjLENBQUMsS0FBSyxDQUFDLE1BQU0sRUFBRTt3QkFDNUQsUUFBUSxDQUFDLElBQUksRUFBRSxJQUFJOzs0QkFDZixJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQzs0QkFDNUMsT0FBTztnQ0FDSCxVQUFVLEVBQUU7b0NBQ1IsTUFBTSxFQUFFO3dDQUNKLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTt3Q0FDM0IsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTO3dDQUN6QixVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7cUNBQzlCO29DQUNELE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRTtvQ0FDL0IsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTO29DQUNwQixJQUFJLEVBQUUsSUFBSSxDQUFDLFVBQVU7aUNBQ3hCO2dDQUNELEVBQUUsRUFBRSxLQUFLLENBQUMsRUFBRSxJQUFJLGNBQWMsRUFBRTtnQ0FDaEMsS0FBSyxRQUFFLElBQUksQ0FBQyxtQkFBbUIsMENBQUUsS0FBSztnQ0FDdEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSzs2QkFDNUMsQ0FBQzt3QkFDTixDQUFDO3dCQUNELGNBQWMsQ0FBQyxJQUFJLEVBQUUsSUFBSTs7NEJBQ3JCLE9BQU87Z0NBQ0gsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVO2dDQUMzQixFQUFFLEVBQUUsS0FBSyxDQUFDLEVBQUUsSUFBSSxjQUFjLEVBQUU7Z0NBQ2hDLEtBQUssUUFBRSxJQUFJLENBQUMsbUJBQW1CLDBDQUFFLEtBQUs7Z0NBQ3RDLE1BQU0sRUFBRSxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTTtnQ0FDN0QsVUFBVSxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSzs2QkFDNUMsQ0FBQzt3QkFDTixDQUFDO3FCQUNKLENBQXFDLENBQUM7aUJBQzFDLENBQUM7WUFDTixDQUFDO1NBQ0osQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOztPQUVHO0lBQ0ssa0JBQWtCLENBQUMsSUFBc0IsRUFBRSxNQUFlO1FBQzlELE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDN0IsSUFBSSxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsRUFBRTtZQUNoRCxNQUFNLElBQUksS0FBSyxDQUFDLHFDQUFxQyxLQUFLLENBQUMsTUFBTSxzQkFBc0IsTUFBTSxTQUFTLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO1NBQ3pIO1FBQ0QsSUFBSSxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRTtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLHNDQUFzQyxLQUFLLENBQUMsT0FBTyxzQkFBc0IsTUFBTSxTQUFTLElBQUksQ0FBQyxPQUFPLEdBQUcsQ0FBQyxDQUFDO1NBQzVIO0lBQ0wsQ0FBQztDQUNKO0FBcE5ELHNCQW9OQztBQUNELFNBQVMsbUJBQW1CLENBQUMsQ0FBcUIsRUFBRSxDQUFTO0lBQ3pELE9BQU8sQ0FBQyxJQUFJLENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQ2xELENBQUM7QUFDRDs7OztHQUlHO0FBQ0gsU0FBUyxjQUFjLENBQUMsT0FBZTtJQUNuQyxJQUFJLE9BQU8sS0FBSyxFQUFFLEVBQUU7UUFDaEIsT0FBTyxVQUFVLENBQUM7S0FDckI7SUFDRCxJQUFJLE9BQU8sS0FBSyxDQUFDLEVBQUU7UUFDZixPQUFPLFVBQVUsQ0FBQztLQUNyQjtJQUNELElBQUksT0FBTyxHQUFHLEVBQUUsRUFBRTtRQUNkLE9BQU8sQ0FBQyxPQUFPLEdBQUcsRUFBRSxDQUFDLEdBQUcsVUFBVSxDQUFDO0tBQ3RDO0lBQ0QsT0FBTyxPQUFPLEdBQUcsVUFBVSxDQUFDO0FBQ2hDLENBQUM7QUFDRCxTQUFTLHVCQUF1QixDQUFDLFNBQWtCO0lBQy9DLElBQUksU0FBUyxLQUFLLFNBQVMsRUFBRTtRQUN6QixPQUFPLFNBQVMsQ0FBQztLQUNwQjtJQUNELE1BQU0sTUFBTSxHQUFHLDBCQUFjLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDekMsSUFBSSxNQUFNLENBQUMsSUFBSSxLQUFLLFFBQVEsRUFBRTtRQUMxQixPQUFPLE1BQU0sQ0FBQyxTQUFTLENBQUM7S0FDM0I7SUFDRCxPQUFPLFNBQVMsQ0FBQztBQUNyQixDQUFDO0FBQ0QsU0FBUyx5QkFBeUIsQ0FBQyxTQUFrQjtJQUNqRCxJQUFJLFNBQVMsS0FBSyxTQUFTLEVBQUU7UUFDekIsT0FBTyxTQUFTLENBQUM7S0FDcEI7SUFDRCxNQUFNLE1BQU0sR0FBRywwQkFBYyxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ3pDLElBQUksTUFBTSxDQUFDLElBQUksS0FBSyxZQUFZLEVBQUU7UUFDOUIsOERBQThEO1FBQzlELG9FQUFvRTtRQUNwRSxPQUFPLFNBQVMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztLQUNsQztJQUNELE9BQU8sU0FBUyxDQUFDO0FBQ3JCLENBQUM7QUFDRCxTQUFTLHFCQUFxQixDQUFDLElBQTRCO0lBQ3ZELE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztBQUNyRCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0LCBJUmVzb3VyY2UsIExhenksIFJlc291cmNlLCBTdGFjaywgVG9rZW4gfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IElBbGFybUFjdGlvbiB9IGZyb20gJy4vYWxhcm0tYWN0aW9uJztcbmltcG9ydCB7IENmbkFsYXJtLCBDZm5BbGFybVByb3BzIH0gZnJvbSAnLi9jbG91ZHdhdGNoLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBIb3Jpem9udGFsQW5ub3RhdGlvbiB9IGZyb20gJy4vZ3JhcGgnO1xuaW1wb3J0IHsgQ3JlYXRlQWxhcm1PcHRpb25zIH0gZnJvbSAnLi9tZXRyaWMnO1xuaW1wb3J0IHsgSU1ldHJpYywgTWV0cmljRXhwcmVzc2lvbkNvbmZpZywgTWV0cmljU3RhdENvbmZpZyB9IGZyb20gJy4vbWV0cmljLXR5cGVzJztcbmltcG9ydCB7IGRpc3BhdGNoTWV0cmljLCBtZXRyaWNQZXJpb2QgfSBmcm9tICcuL3ByaXZhdGUvbWV0cmljLXV0aWwnO1xuaW1wb3J0IHsgZHJvcFVuZGVmaW5lZCB9IGZyb20gJy4vcHJpdmF0ZS9vYmplY3QnO1xuaW1wb3J0IHsgTWV0cmljU2V0IH0gZnJvbSAnLi9wcml2YXRlL3JlbmRlcmluZyc7XG5pbXBvcnQgeyBwYXJzZVN0YXRpc3RpYyB9IGZyb20gJy4vcHJpdmF0ZS9zdGF0aXN0aWMnO1xuLyoqXG4gKiBSZXByZXNlbnRzIGEgQ2xvdWRXYXRjaCBBbGFybVxuICovXG5leHBvcnQgaW50ZXJmYWNlIElBbGFybSBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgLyoqXG4gICAgICogQWxhcm0gQVJOIChpLmUuIGFybjphd3M6Y2xvdWR3YXRjaDo8cmVnaW9uPjo8YWNjb3VudC1pZD46YWxhcm06Rm9vKVxuICAgICAqXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFsYXJtQXJuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogTmFtZSBvZiB0aGUgYWxhcm1cbiAgICAgKlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICByZWFkb25seSBhbGFybU5hbWU6IHN0cmluZztcbn1cbi8qKlxuICogUHJvcGVydGllcyBmb3IgQWxhcm1zXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWxhcm1Qcm9wcyBleHRlbmRzIENyZWF0ZUFsYXJtT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIG1ldHJpYyB0byBhZGQgdGhlIGFsYXJtIG9uXG4gICAgICpcbiAgICAgKiBNZXRyaWMgb2JqZWN0cyBjYW4gYmUgb2J0YWluZWQgZnJvbSBtb3N0IHJlc291cmNlcywgb3IgeW91IGNhbiBjb25zdHJ1Y3RcbiAgICAgKiBjdXN0b20gTWV0cmljIG9iamVjdHMgYnkgaW5zdGFudGlhdGluZyBvbmUuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWV0cmljOiBJTWV0cmljO1xufVxuLyoqXG4gKiBDb21wYXJpc29uIG9wZXJhdG9yIGZvciBldmFsdWF0aW5nIGFsYXJtc1xuICovXG5leHBvcnQgZW51bSBDb21wYXJpc29uT3BlcmF0b3Ige1xuICAgIC8qKlxuICAgICAqIFNwZWNpZmllZCBzdGF0aXN0aWMgaXMgZ3JlYXRlciB0aGFuIG9yIGVxdWFsIHRvIHRoZSB0aHJlc2hvbGRcbiAgICAgKi9cbiAgICBHUkVBVEVSX1RIQU5fT1JfRVFVQUxfVE9fVEhSRVNIT0xEID0gJ0dyZWF0ZXJUaGFuT3JFcXVhbFRvVGhyZXNob2xkJyxcbiAgICAvKipcbiAgICAgKiBTcGVjaWZpZWQgc3RhdGlzdGljIGlzIHN0cmljdGx5IGdyZWF0ZXIgdGhhbiB0aGUgdGhyZXNob2xkXG4gICAgICovXG4gICAgR1JFQVRFUl9USEFOX1RIUkVTSE9MRCA9ICdHcmVhdGVyVGhhblRocmVzaG9sZCcsXG4gICAgLyoqXG4gICAgICogU3BlY2lmaWVkIHN0YXRpc3RpYyBpcyBzdHJpY3RseSBsZXNzIHRoYW4gdGhlIHRocmVzaG9sZFxuICAgICAqL1xuICAgIExFU1NfVEhBTl9USFJFU0hPTEQgPSAnTGVzc1RoYW5UaHJlc2hvbGQnLFxuICAgIC8qKlxuICAgICAqIFNwZWNpZmllZCBzdGF0aXN0aWMgaXMgbGVzcyB0aGFuIG9yIGVxdWFsIHRvIHRoZSB0aHJlc2hvbGQuXG4gICAgICovXG4gICAgTEVTU19USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCA9ICdMZXNzVGhhbk9yRXF1YWxUb1RocmVzaG9sZCcsXG4gICAgLyoqXG4gICAgICogU3BlY2lmaWVkIHN0YXRpc3RpYyBpcyBsb3dlciB0aGFuIG9yIGdyZWF0ZXIgdGhhbiB0aGUgYW5vbWFseSBtb2RlbCBiYW5kLlxuICAgICAqIFVzZWQgb25seSBmb3IgYWxhcm1zIGJhc2VkIG9uIGFub21hbHkgZGV0ZWN0aW9uIG1vZGVsc1xuICAgICAqL1xuICAgIExFU1NfVEhBTl9MT1dFUl9PUl9HUkVBVEVSX1RIQU5fVVBQRVJfVEhSRVNIT0xEID0gJ0xlc3NUaGFuTG93ZXJPckdyZWF0ZXJUaGFuVXBwZXJUaHJlc2hvbGQnXG59XG5jb25zdCBPUEVSQVRPUl9TWU1CT0xTOiB7XG4gICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xufSA9IHtcbiAgICBHcmVhdGVyVGhhbk9yRXF1YWxUb1RocmVzaG9sZDogJz49JyxcbiAgICBHcmVhdGVyVGhhblRocmVzaG9sZDogJz4nLFxuICAgIExlc3NUaGFuVGhyZXNob2xkOiAnPCcsXG4gICAgTGVzc1RoYW5PckVxdWFsVG9UaHJlc2hvbGQ6ICc+PScsXG59O1xuLyoqXG4gKiBTcGVjaWZ5IGhvdyBtaXNzaW5nIGRhdGEgcG9pbnRzIGFyZSB0cmVhdGVkIGR1cmluZyBhbGFybSBldmFsdWF0aW9uXG4gKi9cbmV4cG9ydCBlbnVtIFRyZWF0TWlzc2luZ0RhdGEge1xuICAgIC8qKlxuICAgICAqIE1pc3NpbmcgZGF0YSBwb2ludHMgYXJlIHRyZWF0ZWQgYXMgYnJlYWNoaW5nIHRoZSB0aHJlc2hvbGRcbiAgICAgKi9cbiAgICBCUkVBQ0hJTkcgPSAnYnJlYWNoaW5nJyxcbiAgICAvKipcbiAgICAgKiBNaXNzaW5nIGRhdGEgcG9pbnRzIGFyZSB0cmVhdGVkIGFzIGJlaW5nIHdpdGhpbiB0aGUgdGhyZXNob2xkXG4gICAgICovXG4gICAgTk9UX0JSRUFDSElORyA9ICdub3RCcmVhY2hpbmcnLFxuICAgIC8qKlxuICAgICAqIFRoZSBjdXJyZW50IGFsYXJtIHN0YXRlIGlzIG1haW50YWluZWRcbiAgICAgKi9cbiAgICBJR05PUkUgPSAnaWdub3JlJyxcbiAgICAvKipcbiAgICAgKiBUaGUgYWxhcm0gZG9lcyBub3QgY29uc2lkZXIgbWlzc2luZyBkYXRhIHBvaW50cyB3aGVuIGV2YWx1YXRpbmcgd2hldGhlciB0byBjaGFuZ2Ugc3RhdGVcbiAgICAgKi9cbiAgICBNSVNTSU5HID0gJ21pc3NpbmcnXG59XG4vKipcbiAqIEFuIGFsYXJtIG9uIGEgQ2xvdWRXYXRjaCBtZXRyaWNcbiAqL1xuZXhwb3J0IGNsYXNzIEFsYXJtIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJQWxhcm0ge1xuICAgIC8qKlxuICAgICAqIEltcG9ydCBhbiBleGlzdGluZyBDbG91ZFdhdGNoIGFsYXJtIHByb3ZpZGVkIGFuIEFSTlxuICAgICAqXG4gICAgICogQHBhcmFtIHNjb3BlIFRoZSBwYXJlbnQgY3JlYXRpbmcgY29uc3RydWN0ICh1c3VhbGx5IGB0aGlzYCkuXG4gICAgICogQHBhcmFtIGlkIFRoZSBjb25zdHJ1Y3QncyBuYW1lXG4gICAgICogQHBhcmFtIGFsYXJtQXJuIEFsYXJtIEFSTiAoaS5lLiBhcm46YXdzOmNsb3Vkd2F0Y2g6PHJlZ2lvbj46PGFjY291bnQtaWQ+OmFsYXJtOkZvbylcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21BbGFybUFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhbGFybUFybjogc3RyaW5nKTogSUFsYXJtIHtcbiAgICAgICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJQWxhcm0ge1xuICAgICAgICAgICAgcHVibGljIHJlYWRvbmx5IGFsYXJtQXJuID0gYWxhcm1Bcm47XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgYWxhcm1OYW1lID0gU3RhY2sub2Yoc2NvcGUpLnBhcnNlQXJuKGFsYXJtQXJuLCAnOicpLnJlc291cmNlTmFtZSE7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQVJOIG9mIHRoaXMgYWxhcm1cbiAgICAgKlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgYWxhcm1Bcm46IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBOYW1lIG9mIHRoaXMgYWxhcm0uXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGFsYXJtTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBtZXRyaWMgb2JqZWN0IHRoaXMgYWxhcm0gd2FzIGJhc2VkIG9uXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IG1ldHJpYzogSU1ldHJpYztcbiAgICBwcml2YXRlIGFsYXJtQWN0aW9uQXJucz86IHN0cmluZ1tdO1xuICAgIHByaXZhdGUgaW5zdWZmaWNpZW50RGF0YUFjdGlvbkFybnM/OiBzdHJpbmdbXTtcbiAgICBwcml2YXRlIG9rQWN0aW9uQXJucz86IHN0cmluZ1tdO1xuICAgIC8qKlxuICAgICAqIFRoaXMgbWV0cmljIGFzIGFuIGFubm90YXRpb25cbiAgICAgKi9cbiAgICBwcml2YXRlIHJlYWRvbmx5IGFubm90YXRpb246IEhvcml6b250YWxBbm5vdGF0aW9uO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBbGFybVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgICAgICAgcGh5c2ljYWxOYW1lOiBwcm9wcy5hbGFybU5hbWUsXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBjb21wYXJpc29uT3BlcmF0b3IgPSBwcm9wcy5jb21wYXJpc29uT3BlcmF0b3IgfHwgQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9PUl9FUVVBTF9UT19USFJFU0hPTEQ7XG4gICAgICAgIC8vIFJlbmRlciBtZXRyaWMsIHByb2Nlc3MgcG90ZW50aWFsIG92ZXJyaWRlcyBmcm9tIHRoZSBhbGFybVxuICAgICAgICAvLyAoSXQgd291bGQgYmUgcHJlZmVyYWJsZSBpZiB0aGUgc3RhdGlzdGljIGV0Yy4gd2FzIHdvcmtlZCBpbnRvIHRoZSBtZXRyaWMsXG4gICAgICAgIC8vIGJ1dCBoZXkgd2UncmUgYWxsb3dpbmcgb3ZlcnJpZGVzLi4uKVxuICAgICAgICBjb25zdCBtZXRyaWNQcm9wczogV3JpdGVhYmxlPFBhcnRpYWw8Q2ZuQWxhcm1Qcm9wcz4+ID0gdGhpcy5yZW5kZXJNZXRyaWMocHJvcHMubWV0cmljKTtcbiAgICAgICAgaWYgKHByb3BzLnBlcmlvZCkge1xuICAgICAgICAgICAgbWV0cmljUHJvcHMucGVyaW9kID0gcHJvcHMucGVyaW9kLnRvU2Vjb25kcygpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5zdGF0aXN0aWMpIHtcbiAgICAgICAgICAgIC8vIFdpbGwgb3ZlcndyaXRlIGJvdGggZmllbGRzIGlmIHByZXNlbnRcbiAgICAgICAgICAgIE9iamVjdC5hc3NpZ24obWV0cmljUHJvcHMsIHtcbiAgICAgICAgICAgICAgICBzdGF0aXN0aWM6IHJlbmRlcklmU2ltcGxlU3RhdGlzdGljKHByb3BzLnN0YXRpc3RpYyksXG4gICAgICAgICAgICAgICAgZXh0ZW5kZWRTdGF0aXN0aWM6IHJlbmRlcklmRXh0ZW5kZWRTdGF0aXN0aWMocHJvcHMuc3RhdGlzdGljKSxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGFsYXJtID0gbmV3IENmbkFsYXJtKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIC8vIE1ldGFcbiAgICAgICAgICAgIGFsYXJtRGVzY3JpcHRpb246IHByb3BzLmFsYXJtRGVzY3JpcHRpb24sXG4gICAgICAgICAgICBhbGFybU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgICAgICAgLy8gRXZhbHVhdGlvblxuICAgICAgICAgICAgY29tcGFyaXNvbk9wZXJhdG9yLFxuICAgICAgICAgICAgdGhyZXNob2xkOiBwcm9wcy50aHJlc2hvbGQsXG4gICAgICAgICAgICBkYXRhcG9pbnRzVG9BbGFybTogcHJvcHMuZGF0YXBvaW50c1RvQWxhcm0sXG4gICAgICAgICAgICBldmFsdWF0ZUxvd1NhbXBsZUNvdW50UGVyY2VudGlsZTogcHJvcHMuZXZhbHVhdGVMb3dTYW1wbGVDb3VudFBlcmNlbnRpbGUsXG4gICAgICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogcHJvcHMuZXZhbHVhdGlvblBlcmlvZHMsXG4gICAgICAgICAgICB0cmVhdE1pc3NpbmdEYXRhOiBwcm9wcy50cmVhdE1pc3NpbmdEYXRhLFxuICAgICAgICAgICAgLy8gQWN0aW9uc1xuICAgICAgICAgICAgYWN0aW9uc0VuYWJsZWQ6IHByb3BzLmFjdGlvbnNFbmFibGVkLFxuICAgICAgICAgICAgYWxhcm1BY3Rpb25zOiBMYXp5Lmxpc3RWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuYWxhcm1BY3Rpb25Bcm5zIH0pLFxuICAgICAgICAgICAgaW5zdWZmaWNpZW50RGF0YUFjdGlvbnM6IExhenkubGlzdFZhbHVlKHsgcHJvZHVjZTogKCgpID0+IHRoaXMuaW5zdWZmaWNpZW50RGF0YUFjdGlvbkFybnMpIH0pLFxuICAgICAgICAgICAgb2tBY3Rpb25zOiBMYXp5Lmxpc3RWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMub2tBY3Rpb25Bcm5zIH0pLFxuICAgICAgICAgICAgLy8gTWV0cmljXG4gICAgICAgICAgICAuLi5tZXRyaWNQcm9wcyxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuYWxhcm1Bcm4gPSB0aGlzLmdldFJlc291cmNlQXJuQXR0cmlidXRlKGFsYXJtLmF0dHJBcm4sIHtcbiAgICAgICAgICAgIHNlcnZpY2U6ICdjbG91ZHdhdGNoJyxcbiAgICAgICAgICAgIHJlc291cmNlOiAnYWxhcm0nLFxuICAgICAgICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgICAgICAgIHNlcDogJzonLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5hbGFybU5hbWUgPSB0aGlzLmdldFJlc291cmNlTmFtZUF0dHJpYnV0ZShhbGFybS5yZWYpO1xuICAgICAgICB0aGlzLm1ldHJpYyA9IHByb3BzLm1ldHJpYztcbiAgICAgICAgY29uc3QgZGF0YXBvaW50cyA9IHByb3BzLmRhdGFwb2ludHNUb0FsYXJtIHx8IHByb3BzLmV2YWx1YXRpb25QZXJpb2RzO1xuICAgICAgICB0aGlzLmFubm90YXRpb24gPSB7XG4gICAgICAgICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgICAgICAgICBsYWJlbDogYCR7dGhpcy5tZXRyaWN9ICR7T1BFUkFUT1JfU1lNQk9MU1tjb21wYXJpc29uT3BlcmF0b3JdfSAke3Byb3BzLnRocmVzaG9sZH0gZm9yICR7ZGF0YXBvaW50c30gZGF0YXBvaW50cyB3aXRoaW4gJHtkZXNjcmliZVBlcmlvZChwcm9wcy5ldmFsdWF0aW9uUGVyaW9kcyAqIG1ldHJpY1BlcmlvZChwcm9wcy5tZXRyaWMpLnRvU2Vjb25kcygpKX1gLFxuICAgICAgICAgICAgdmFsdWU6IHByb3BzLnRocmVzaG9sZCxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogVHJpZ2dlciB0aGlzIGFjdGlvbiBpZiB0aGUgYWxhcm0gZmlyZXNcbiAgICAgKlxuICAgICAqIFR5cGljYWxseSB0aGUgQVJOIG9mIGFuIFNOUyB0b3BpYyBvciBBUk4gb2YgYW4gQXV0b1NjYWxpbmcgcG9saWN5LlxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRBbGFybUFjdGlvbiguLi5hY3Rpb25zOiBJQWxhcm1BY3Rpb25bXSkge1xuICAgICAgICBpZiAodGhpcy5hbGFybUFjdGlvbkFybnMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhpcy5hbGFybUFjdGlvbkFybnMgPSBbXTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmFsYXJtQWN0aW9uQXJucy5wdXNoKC4uLmFjdGlvbnMubWFwKGEgPT4gYS5iaW5kKHRoaXMsIHRoaXMpLmFsYXJtQWN0aW9uQXJuKSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRyaWdnZXIgdGhpcyBhY3Rpb24gaWYgdGhlcmUgaXMgaW5zdWZmaWNpZW50IGRhdGEgdG8gZXZhbHVhdGUgdGhlIGFsYXJtXG4gICAgICpcbiAgICAgKiBUeXBpY2FsbHkgdGhlIEFSTiBvZiBhbiBTTlMgdG9waWMgb3IgQVJOIG9mIGFuIEF1dG9TY2FsaW5nIHBvbGljeS5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkSW5zdWZmaWNpZW50RGF0YUFjdGlvbiguLi5hY3Rpb25zOiBJQWxhcm1BY3Rpb25bXSkge1xuICAgICAgICBpZiAodGhpcy5pbnN1ZmZpY2llbnREYXRhQWN0aW9uQXJucyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB0aGlzLmluc3VmZmljaWVudERhdGFBY3Rpb25Bcm5zID0gW107XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5pbnN1ZmZpY2llbnREYXRhQWN0aW9uQXJucy5wdXNoKC4uLmFjdGlvbnMubWFwKGEgPT4gYS5iaW5kKHRoaXMsIHRoaXMpLmFsYXJtQWN0aW9uQXJuKSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRyaWdnZXIgdGhpcyBhY3Rpb24gaWYgdGhlIGFsYXJtIHJldHVybnMgZnJvbSBicmVhY2hpbmcgc3RhdGUgaW50byBvayBzdGF0ZVxuICAgICAqXG4gICAgICogVHlwaWNhbGx5IHRoZSBBUk4gb2YgYW4gU05TIHRvcGljIG9yIEFSTiBvZiBhbiBBdXRvU2NhbGluZyBwb2xpY3kuXG4gICAgICovXG4gICAgcHVibGljIGFkZE9rQWN0aW9uKC4uLmFjdGlvbnM6IElBbGFybUFjdGlvbltdKSB7XG4gICAgICAgIGlmICh0aGlzLm9rQWN0aW9uQXJucyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB0aGlzLm9rQWN0aW9uQXJucyA9IFtdO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMub2tBY3Rpb25Bcm5zLnB1c2goLi4uYWN0aW9ucy5tYXAoYSA9PiBhLmJpbmQodGhpcywgdGhpcykuYWxhcm1BY3Rpb25Bcm4pKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogVHVybiB0aGlzIGFsYXJtIGludG8gYSBob3Jpem9udGFsIGFubm90YXRpb25cbiAgICAgKlxuICAgICAqIFRoaXMgaXMgdXNlZnVsIGlmIHlvdSB3YW50IHRvIHJlcHJlc2VudCBhbiBBbGFybSBpbiBhIG5vbi1BbGFybVdpZGdldC5cbiAgICAgKiBBbiBgQWxhcm1XaWRnZXRgIGNhbiBkaXJlY3RseSBzaG93IGFuIGFsYXJtLCBidXQgaXQgY2FuIG9ubHkgc2hvdyBhXG4gICAgICogc2luZ2xlIGFsYXJtIGFuZCBubyBvdGhlciBtZXRyaWNzLiBJbnN0ZWFkLCB5b3UgY2FuIGNvbnZlcnQgdGhlIGFsYXJtIHRvXG4gICAgICogYSBIb3Jpem9udGFsQW5ub3RhdGlvbiBhbmQgYWRkIGl0IGFzIGFuIGFubm90YXRpb24gdG8gYW5vdGhlciBncmFwaC5cbiAgICAgKlxuICAgICAqIFRoaXMgbWlnaHQgYmUgdXNlZnVsIGlmOlxuICAgICAqXG4gICAgICogLSBZb3Ugd2FudCB0byBzaG93IG11bHRpcGxlIGFsYXJtcyBpbnNpZGUgYSBzaW5nbGUgZ3JhcGgsIGZvciBleGFtcGxlIGlmXG4gICAgICogICB5b3UgaGF2ZSBib3RoIGEgXCJzbWFsbCBtYXJnaW4vbG9uZyBwZXJpb2RcIiBhbGFybSBhcyB3ZWxsIGFzIGFcbiAgICAgKiAgIFwibGFyZ2UgbWFyZ2luL3Nob3J0IHBlcmlvZFwiIGFsYXJtLlxuICAgICAqXG4gICAgICogLSBZb3Ugd2FudCB0byBzaG93IGFuIEFsYXJtIGxpbmUgaW4gYSBncmFwaCB3aXRoIG11bHRpcGxlIG1ldHJpY3MgaW4gaXQuXG4gICAgICovXG4gICAgcHVibGljIHRvQW5ub3RhdGlvbigpOiBIb3Jpem9udGFsQW5ub3RhdGlvbiB7XG4gICAgICAgIHJldHVybiB0aGlzLmFubm90YXRpb247XG4gICAgfVxuICAgIHByaXZhdGUgcmVuZGVyTWV0cmljKG1ldHJpYzogSU1ldHJpYykge1xuICAgICAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICAgICAgcmV0dXJuIGRpc3BhdGNoTWV0cmljKG1ldHJpYywge1xuICAgICAgICAgICAgd2l0aFN0YXQoc3QpIHtcbiAgICAgICAgICAgICAgICBzZWxmLnZhbGlkYXRlTWV0cmljU3RhdChzdCwgbWV0cmljKTtcbiAgICAgICAgICAgICAgICByZXR1cm4gZHJvcFVuZGVmaW5lZCh7XG4gICAgICAgICAgICAgICAgICAgIGRpbWVuc2lvbnM6IHN0LmRpbWVuc2lvbnMsXG4gICAgICAgICAgICAgICAgICAgIG5hbWVzcGFjZTogc3QubmFtZXNwYWNlLFxuICAgICAgICAgICAgICAgICAgICBtZXRyaWNOYW1lOiBzdC5tZXRyaWNOYW1lLFxuICAgICAgICAgICAgICAgICAgICBwZXJpb2Q6IHN0LnBlcmlvZD8udG9TZWNvbmRzKCksXG4gICAgICAgICAgICAgICAgICAgIHN0YXRpc3RpYzogcmVuZGVySWZTaW1wbGVTdGF0aXN0aWMoc3Quc3RhdGlzdGljKSxcbiAgICAgICAgICAgICAgICAgICAgZXh0ZW5kZWRTdGF0aXN0aWM6IHJlbmRlcklmRXh0ZW5kZWRTdGF0aXN0aWMoc3Quc3RhdGlzdGljKSxcbiAgICAgICAgICAgICAgICAgICAgdW5pdDogc3QudW5pdEZpbHRlcixcbiAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB3aXRoRXhwcmVzc2lvbigpIHtcbiAgICAgICAgICAgICAgICAvLyBFeHBhbmQgdGhlIG1hdGggZXhwcmVzc2lvbiBtZXRyaWMgaW50byBhIHNldFxuICAgICAgICAgICAgICAgIGNvbnN0IG1zZXQgPSBuZXcgTWV0cmljU2V0PGJvb2xlYW4+KCk7XG4gICAgICAgICAgICAgICAgbXNldC5hZGRUb3BMZXZlbCh0cnVlLCBtZXRyaWMpO1xuICAgICAgICAgICAgICAgIGxldCBlaWQgPSAwO1xuICAgICAgICAgICAgICAgIGZ1bmN0aW9uIHVuaXF1ZU1ldHJpY0lkKCkge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gYGV4cHJfJHsrK2VpZH1gO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgICAgICAgICBtZXRyaWNzOiBtc2V0LmVudHJpZXMubWFwKGVudHJ5ID0+IGRpc3BhdGNoTWV0cmljKGVudHJ5Lm1ldHJpYywge1xuICAgICAgICAgICAgICAgICAgICAgICAgd2l0aFN0YXQoc3RhdCwgY29uZikge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHNlbGYudmFsaWRhdGVNZXRyaWNTdGF0KHN0YXQsIGVudHJ5Lm1ldHJpYyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgbWV0cmljU3RhdDoge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgbWV0cmljOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgbWV0cmljTmFtZTogc3RhdC5tZXRyaWNOYW1lLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIG5hbWVzcGFjZTogc3RhdC5uYW1lc3BhY2UsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgZGltZW5zaW9uczogc3RhdC5kaW1lbnNpb25zLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIHBlcmlvZDogc3RhdC5wZXJpb2QudG9TZWNvbmRzKCksXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBzdGF0OiBzdGF0LnN0YXRpc3RpYyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIHVuaXQ6IHN0YXQudW5pdEZpbHRlcixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgaWQ6IGVudHJ5LmlkIHx8IHVuaXF1ZU1ldHJpY0lkKCksXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGxhYmVsOiBjb25mLnJlbmRlcmluZ1Byb3BlcnRpZXM/LmxhYmVsLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICByZXR1cm5EYXRhOiBlbnRyeS50YWcgPyB1bmRlZmluZWQgOiBmYWxzZSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgIHdpdGhFeHByZXNzaW9uKGV4cHIsIGNvbmYpIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBleHByZXNzaW9uOiBleHByLmV4cHJlc3Npb24sXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGlkOiBlbnRyeS5pZCB8fCB1bmlxdWVNZXRyaWNJZCgpLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBsYWJlbDogY29uZi5yZW5kZXJpbmdQcm9wZXJ0aWVzPy5sYWJlbCxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgcGVyaW9kOiBtYXRoRXhwckhhc1N1Ym1ldHJpY3MoZXhwcikgPyB1bmRlZmluZWQgOiBleHByLnBlcmlvZCxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgcmV0dXJuRGF0YTogZW50cnkudGFnID8gdW5kZWZpbmVkIDogZmFsc2UsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgfTtcbiAgICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIH0pIGFzIENmbkFsYXJtLk1ldHJpY0RhdGFRdWVyeVByb3BlcnR5KSxcbiAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFZhbGlkYXRlIHRoYXQgaWYgYSByZWdpb24gYW5kIGFjY291bnQgYXJlIGluIHRoZSBnaXZlbiBzdGF0IGNvbmZpZywgdGhleSBtYXRjaCB0aGUgQWxhcm1cbiAgICAgKi9cbiAgICBwcml2YXRlIHZhbGlkYXRlTWV0cmljU3RhdChzdGF0OiBNZXRyaWNTdGF0Q29uZmlnLCBtZXRyaWM6IElNZXRyaWMpIHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcbiAgICAgICAgaWYgKGRlZmluaXRlbHlEaWZmZXJlbnQoc3RhdC5yZWdpb24sIHN0YWNrLnJlZ2lvbikpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgQ2Fubm90IGNyZWF0ZSBhbiBBbGFybSBpbiByZWdpb24gJyR7c3RhY2sucmVnaW9ufScgYmFzZWQgb24gbWV0cmljICcke21ldHJpY30nIGluICcke3N0YXQucmVnaW9ufSdgKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoZGVmaW5pdGVseURpZmZlcmVudChzdGF0LmFjY291bnQsIHN0YWNrLmFjY291bnQpKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBjcmVhdGUgYW4gQWxhcm0gaW4gYWNjb3VudCAnJHtzdGFjay5hY2NvdW50fScgYmFzZWQgb24gbWV0cmljICcke21ldHJpY30nIGluICcke3N0YXQuYWNjb3VudH0nYCk7XG4gICAgICAgIH1cbiAgICB9XG59XG5mdW5jdGlvbiBkZWZpbml0ZWx5RGlmZmVyZW50KHg6IHN0cmluZyB8IHVuZGVmaW5lZCwgeTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHggJiYgIVRva2VuLmlzVW5yZXNvbHZlZCh5KSAmJiB4ICE9PSB5O1xufVxuLyoqXG4gKiBSZXR1cm4gYSBodW1hbiByZWFkYWJsZSBzdHJpbmcgZm9yIHRoaXMgcGVyaW9kXG4gKlxuICogV2Uga25vdyB0aGUgc2Vjb25kcyBhcmUgYWx3YXlzIG9uZSBvZiBhIGhhbmRmdWwgb2YgYWxsb3dlZCB2YWx1ZXMuXG4gKi9cbmZ1bmN0aW9uIGRlc2NyaWJlUGVyaW9kKHNlY29uZHM6IG51bWJlcikge1xuICAgIGlmIChzZWNvbmRzID09PSA2MCkge1xuICAgICAgICByZXR1cm4gJzEgbWludXRlJztcbiAgICB9XG4gICAgaWYgKHNlY29uZHMgPT09IDEpIHtcbiAgICAgICAgcmV0dXJuICcxIHNlY29uZCc7XG4gICAgfVxuICAgIGlmIChzZWNvbmRzID4gNjApIHtcbiAgICAgICAgcmV0dXJuIChzZWNvbmRzIC8gNjApICsgJyBtaW51dGVzJztcbiAgICB9XG4gICAgcmV0dXJuIHNlY29uZHMgKyAnIHNlY29uZHMnO1xufVxuZnVuY3Rpb24gcmVuZGVySWZTaW1wbGVTdGF0aXN0aWMoc3RhdGlzdGljPzogc3RyaW5nKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoc3RhdGlzdGljID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgY29uc3QgcGFyc2VkID0gcGFyc2VTdGF0aXN0aWMoc3RhdGlzdGljKTtcbiAgICBpZiAocGFyc2VkLnR5cGUgPT09ICdzaW1wbGUnKSB7XG4gICAgICAgIHJldHVybiBwYXJzZWQuc3RhdGlzdGljO1xuICAgIH1cbiAgICByZXR1cm4gdW5kZWZpbmVkO1xufVxuZnVuY3Rpb24gcmVuZGVySWZFeHRlbmRlZFN0YXRpc3RpYyhzdGF0aXN0aWM/OiBzdHJpbmcpOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICAgIGlmIChzdGF0aXN0aWMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICBjb25zdCBwYXJzZWQgPSBwYXJzZVN0YXRpc3RpYyhzdGF0aXN0aWMpO1xuICAgIGlmIChwYXJzZWQudHlwZSA9PT0gJ3BlcmNlbnRpbGUnKSB7XG4gICAgICAgIC8vIEFscmVhZHkgcGVyY2VudGlsZS4gQXZvaWQgcGFyc2luZyBiZWNhdXNlIHdlIG1pZ2h0IGdldCBpbnRvXG4gICAgICAgIC8vIGZsb2F0aW5nIHBvaW50IHJvdW5kaW5nIGlzc3VlcywgcmV0dXJuIGFzLWlzIGJ1dCBsb3dlcmNhc2UgdGhlIHAuXG4gICAgICAgIHJldHVybiBzdGF0aXN0aWMudG9Mb3dlckNhc2UoKTtcbiAgICB9XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbn1cbmZ1bmN0aW9uIG1hdGhFeHBySGFzU3VibWV0cmljcyhleHByOiBNZXRyaWNFeHByZXNzaW9uQ29uZmlnKSB7XG4gICAgcmV0dXJuIE9iamVjdC5rZXlzKGV4cHIudXNpbmdNZXRyaWNzKS5sZW5ndGggPiAwO1xufVxudHlwZSBXcml0ZWFibGU8VD4gPSB7XG4gICAgLXJlYWRvbmx5IFtQIGluIGtleW9mIFRdOiBUW1BdO1xufTtcbiJdfQ==