"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error('Exactly one of \'customMetric\' or \'predefinedMetric\' must be specified.');
        }
        if (props.predefinedMetric === PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET && !props.resourceLabel) {
            throw new Error('When tracking the ALBRequestCountPerTarget metric, the ALB identifier must be supplied in resourceLabel');
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error('Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.');
        }
        super(scope, id);
        this.resource = new autoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyType: 'TargetTrackingScaling',
            autoScalingGroupName: props.autoScalingGroup.autoScalingGroupName,
            cooldown: props.cooldown && props.cooldown.toSeconds().toString(),
            estimatedInstanceWarmup: props.estimatedInstanceWarmup && props.estimatedInstanceWarmup.toSeconds(),
            targetTrackingConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                targetValue: props.targetValue,
            },
        });
        this.scalingPolicyArn = this.resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    /**
     * Average CPU utilization of the Auto Scaling group
     */
    PredefinedMetric["ASG_AVERAGE_CPU_UTILIZATION"] = "ASGAverageCPUUtilization";
    /**
     * Average number of bytes received on all network interfaces by the Auto Scaling group
     */
    PredefinedMetric["ASG_AVERAGE_NETWORK_IN"] = "ASGAverageNetworkIn";
    /**
     * Average number of bytes sent out on all network interfaces by the Auto Scaling group
     */
    PredefinedMetric["ASG_AVERAGE_NETWORK_OUT"] = "ASGAverageNetworkOut";
    /**
     * Number of requests completed per target in an Application Load Balancer target group
     *
     * Specify the ALB to look at in the `resourceLabel` field.
     */
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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