"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Arbitrary (valid) array of intervals
 *
 * There are many invalid combinations of interval arrays, so we have
 * to be very specific about generating arrays that are valid. We do this
 * by taking a full, valid interval schedule and progressively stripping parts
 * away from it.
 *
 * Some of the changes may change its meaning, but we take care to never leave
 * a schedule with insufficient information so that the parser will error out.
 */
function generateArbitraryIntervals(mrng) {
    const ret = new Array();
    const absolute = mrng.nextBoolean();
    // Ascending or descending scaling
    const factor = (mrng.nextBoolean() ? 1 : -1) * (absolute ? 10 : 1);
    const bias = absolute ? 50 : 0;
    // Begin with a full schedule
    ret.push({ lower: 0, upper: 10, change: -2 * factor + bias });
    ret.push({ lower: 10, upper: 20, change: -1 * factor + bias });
    ret.push({ lower: 20, upper: 60, change: 0 + bias });
    ret.push({ lower: 60, upper: 80, change: 0 + bias });
    ret.push({ lower: 80, upper: 90, change: 1 * factor + bias });
    ret.push({ lower: 90, upper: Infinity, change: 2 * factor + bias });
    // Take away parts from this. First we see if we do something to the 0-change alarms.
    // The actions can be: remove it OR turn it into a regular change value.
    const noChanges = ret.filter(x => x.change === bias);
    if (!absolute) {
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[0]), 1);
            }
            else {
                noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            }
        }
        if (mrng.nextBoolean()) {
            if (mrng.nextBoolean()) {
                ret.splice(ret.indexOf(noChanges[1]), 1);
            }
            else {
                noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
            }
        }
    }
    else {
        // In absolute mode both have to get the same treatment at the same time
        // otherwise we'll end up with a timeline with two gaps
        if (mrng.nextBoolean()) {
            ret.splice(ret.indexOf(noChanges[0]), 1);
            ret.splice(ret.indexOf(noChanges[1]), 1);
        }
        else {
            noChanges[0] = { ...noChanges[0], change: -1 * factor + bias };
            noChanges[1] = { ...noChanges[1], change: 1 * factor + bias };
        }
    }
    // We might also take away either the bottom or the upper half
    if (mrng.nextInt(0, 2) === 0) {
        const signToStrip = mrng.nextBoolean() ? -1 : 1;
        let ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        while (ix >= 0) {
            ret.splice(ix, 1);
            ix = ret.findIndex(x => Math.sign(x.change - bias) === signToStrip);
        }
    }
    // Then we're going to arbitrarily get rid of bounds in the most naive way possible
    const iterations = mrng.nextInt(0, 10);
    for (let iter = 0; iter < iterations; iter++) {
        const i = mrng.nextInt(0, ret.length - 1);
        if (mrng.nextBoolean()) {
            // scrap lower bound
            // okay if current interval has an upper bound AND the preceding interval has an upper bound
            if (ret[i].upper !== undefined && (i === 0 || ret[i - 1].upper !== undefined)) {
                ret[i] = { ...ret[i], lower: undefined };
            }
        }
        else {
            // scrap upper bound
            // okay if current interval has a lower bound AND the succeeding interval has a lower bound
            if (ret[i].lower !== undefined && (i === ret.length - 1 || ret[i + 1].lower !== undefined)) {
                ret[i] = { ...ret[i], upper: undefined };
            }
        }
    }
    return { absolute, intervals: ret };
}
exports.generateArbitraryIntervals = generateArbitraryIntervals;
//# sourceMappingURL=data:application/json;base64,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