"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * VirtualService represents a service inside an AppMesh
 *
 * It routes traffic either to a Virtual Node or to a Virtual Router.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_services.html
 */
class VirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualServiceName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        if (props.virtualNode && props.virtualRouter) {
            throw new Error('Must provide only one of virtualNode or virtualRouter for the provider');
        }
        this.mesh = props.mesh;
        // Check which provider to use node or router (or neither)
        if (props.virtualRouter) {
            this.virtualServiceProvider = this.addVirtualRouter(props.virtualRouter.virtualRouterName);
        }
        if (props.virtualNode) {
            this.virtualServiceProvider = this.addVirtualNode(props.virtualNode.virtualNodeName);
        }
        const svc = new appmesh_generated_1.CfnVirtualService(this, 'Resource', {
            meshName: this.mesh.meshName,
            virtualServiceName: this.physicalName,
            spec: {
                provider: this.virtualServiceProvider,
            },
        });
        this.virtualServiceName = this.getResourceNameAttribute(svc.attrVirtualServiceName);
        this.virtualServiceArn = this.getResourceArnAttribute(svc.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualService`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualService given an ARN
     */
    static fromVirtualServiceArn(scope, id, virtualServiceArn) {
        return new ImportedVirtualService(scope, id, {
            virtualServiceArn,
        });
    }
    /**
     * Import an existing VirtualService given mesh and service names
     */
    static fromVirtualServiceName(scope, id, meshName, virtualServiceName) {
        return new ImportedVirtualService(scope, id, {
            meshName,
            virtualServiceName,
        });
    }
    addVirtualRouter(name) {
        return {
            virtualRouter: {
                virtualRouterName: name,
            },
        };
    }
    addVirtualNode(name) {
        return {
            virtualNode: {
                virtualNodeName: name,
            },
        };
    }
}
exports.VirtualService = VirtualService;
/**
 * Returns properties that allows a VirtualService to be imported
 */
class ImportedVirtualService extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualServiceArn) {
            this.virtualServiceArn = props.virtualServiceArn;
            this.virtualServiceName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualServiceArn).resourceName));
        }
        else if (props.virtualServiceName && props.meshName) {
            this.virtualServiceName = props.virtualServiceName;
            this.virtualServiceArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualService`,
                resourceName: this.virtualServiceName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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