"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const virtual_node_1 = require("./virtual-node");
const virtual_router_1 = require("./virtual-router");
const virtual_service_1 = require("./virtual-service");
/**
 * A utility enum defined for the egressFilter type property, the default of DROP_ALL,
 * allows traffic only to other resources inside the mesh, or API calls to amazon resources.
 *
 * @default DROP_ALL
 */
var MeshFilterType;
(function (MeshFilterType) {
    /**
     * Allows all outbound traffic
     */
    MeshFilterType["ALLOW_ALL"] = "ALLOW_ALL";
    /**
     * Allows traffic only to other resources inside the mesh, or API calls to amazon resources
     */
    MeshFilterType["DROP_ALL"] = "DROP_ALL";
})(MeshFilterType = exports.MeshFilterType || (exports.MeshFilterType = {}));
/**
 * Represents a new or imported AppMesh mesh
 */
class MeshBase extends cdk.Resource {
    /**
     * Adds a VirtualRouter to the Mesh with the given id and props
     */
    addVirtualRouter(id, props = {}) {
        return new virtual_router_1.VirtualRouter(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * Adds a VirtualService with the given id
     */
    addVirtualService(id, props = {}) {
        return new virtual_service_1.VirtualService(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * Adds a VirtualNode to the Mesh
     */
    addVirtualNode(id, props = {}) {
        return new virtual_node_1.VirtualNode(this, id, {
            ...props,
            mesh: this,
        });
    }
}
/**
 * Define a new AppMesh mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
 */
class Mesh extends MeshBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.meshName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        const mesh = new appmesh_generated_1.CfnMesh(this, 'Resource', {
            meshName: this.physicalName,
            spec: {
                egressFilter: props.egressFilter ? {
                    type: props.egressFilter,
                } : undefined,
            },
        });
        this.meshName = this.getResourceNameAttribute(mesh.attrMeshName);
        this.meshArn = this.getResourceArnAttribute(mesh.ref, {
            service: 'appmesh',
            resource: 'mesh',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing mesh by arn
     */
    static fromMeshArn(scope, id, meshArn) {
        const parts = cdk.Stack.of(scope).parseArn(meshArn);
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = parts.resourceName || '';
                this.meshArn = meshArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import an existing mesh by name
     */
    static fromMeshName(scope, id, meshName) {
        const arn = cdk.Stack.of(scope).formatArn({
            service: 'appmesh',
            resource: 'mesh',
            resourceName: meshName,
        });
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = meshName;
                this.meshArn = arn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Mesh = Mesh;
//# sourceMappingURL=data:application/json;base64,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