"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const url_1 = require("url");
const jsonSchema = require("./json-schema");
exports.ALL_METHODS = ['OPTIONS', 'GET', 'PUT', 'POST', 'DELETE', 'PATCH', 'HEAD'];
const ALLOWED_METHODS = ['ANY', ...exports.ALL_METHODS];
function validateHttpMethod(method, messagePrefix = '') {
    if (!ALLOWED_METHODS.includes(method)) {
        throw new Error(`${messagePrefix}Invalid HTTP method "${method}". Allowed methods: ${ALLOWED_METHODS.join(',')}`);
    }
}
exports.validateHttpMethod = validateHttpMethod;
function parseMethodOptionsPath(originalPath) {
    if (!originalPath.startsWith('/')) {
        throw new Error(`Method options path must start with '/': ${originalPath}`);
    }
    const path = originalPath.substr(1); // trim trailing '/'
    const components = path.split('/');
    if (components.length < 2) {
        throw new Error(`Method options path must include at least two components: /{resource}/{method} (i.e. /foo/bar/GET): ${path}`);
    }
    const httpMethod = components.pop().toUpperCase(); // last component is an HTTP method
    if (httpMethod !== '*') {
        validateHttpMethod(httpMethod, `${originalPath}: `);
    }
    let resourcePath = '/~1' + components.join('~1');
    if (components.length === 1 && components[0] === '*') {
        resourcePath = '/*';
    }
    else if (components.length === 1 && components[0] === '') {
        resourcePath = '/';
    }
    return {
        httpMethod,
        resourcePath,
    };
}
exports.parseMethodOptionsPath = parseMethodOptionsPath;
function parseAwsApiCall(path, action, actionParams) {
    if (actionParams && !action) {
        throw new Error('"actionParams" requires that "action" will be set');
    }
    if (path && action) {
        throw new Error(`"path" and "action" are mutually exclusive (path="${path}", action="${action}")`);
    }
    if (path) {
        return {
            apiType: 'path',
            apiValue: path,
        };
    }
    if (action) {
        if (actionParams) {
            action += '&' + url_1.format({ query: actionParams }).substr(1);
        }
        return {
            apiType: 'action',
            apiValue: action,
        };
    }
    throw new Error('Either "path" or "action" are required');
}
exports.parseAwsApiCall = parseAwsApiCall;
function validateInteger(property, messagePrefix) {
    if (property && !Number.isInteger(property)) {
        throw new Error(`${messagePrefix} should be an integer`);
    }
}
exports.validateInteger = validateInteger;
class JsonSchemaMapper {
    /**
     * Transforms naming of some properties to prefix with a $, where needed
     * according to the JSON schema spec
     * @param schema The JsonSchema object to transform for CloudFormation output
     */
    static toCfnJsonSchema(schema) {
        const result = JsonSchemaMapper._toCfnJsonSchema(schema);
        if (!('$schema' in result)) {
            result.$schema = jsonSchema.JsonSchemaVersion.DRAFT4;
        }
        return result;
    }
    static _toCfnJsonSchema(schema, preserveKeys = false) {
        if (schema == null || typeof schema !== 'object') {
            return schema;
        }
        if (Array.isArray(schema)) {
            return schema.map(entry => JsonSchemaMapper._toCfnJsonSchema(entry));
        }
        return Object.assign({}, ...Object.entries(schema).map(([key, value]) => {
            const mapKey = !preserveKeys && (key in JsonSchemaMapper.SchemaPropsWithPrefix);
            const newKey = mapKey ? JsonSchemaMapper.SchemaPropsWithPrefix[key] : key;
            // If keys were preserved, don't consider SchemaPropsWithUserDefinedChildren for those keys (they are user-defined!)
            const newValue = JsonSchemaMapper._toCfnJsonSchema(value, !preserveKeys && JsonSchemaMapper.SchemaPropsWithUserDefinedChildren[key]);
            return { [newKey]: newValue };
        }));
    }
}
exports.JsonSchemaMapper = JsonSchemaMapper;
JsonSchemaMapper.SchemaPropsWithPrefix = {
    schema: '$schema',
    ref: '$ref',
    id: '$id',
};
// The value indicates whether direct children should be key-mapped.
JsonSchemaMapper.SchemaPropsWithUserDefinedChildren = {
    definitions: true,
    properties: true,
    patternProperties: true,
    dependencies: true,
};
//# sourceMappingURL=data:application/json;base64,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