"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
class RestApiBase extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.restApiName || id,
        });
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Gets the "execute-api" ARN
     * @returns The "execute-api" ARN.
     * @default "*" returns the execute API ARN for all methods/resources in
     * this API.
     * @param method The method (default `*`)
     * @param path The resource path. Must start with '/' (default `*`)
     * @param stage The stage (default `*`)
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
}
/**
 * Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 *
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends RestApiBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi
         */
        this.methods = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this.configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    static fromRestApiId(scope, id, restApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Add an ApiKey
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
    configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApi = RestApi;
var ApiKeySourceType;
(function (ApiKeySourceType) {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    ApiKeySourceType["HEADER"] = "HEADER";
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional API and its custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
    /**
     * For a private API and its custom domain name.
     */
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.restApi = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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