"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const cors_1 = require("./cors");
const integrations_1 = require("./integrations");
const method_1 = require("./method");
class ResourceBase extends core_1.Resource {
    constructor(scope, id) {
        super(scope, id);
        this.children = {};
    }
    addResource(pathPart, options) {
        return new Resource(this, pathPart, { parent: this, pathPart, ...options });
    }
    addMethod(httpMethod, integration, options) {
        return new method_1.Method(this, httpMethod, { resource: this, httpMethod, integration, options });
    }
    addProxy(options) {
        return new ProxyResource(this, '{proxy+}', { parent: this, ...options });
    }
    addCorsPreflight(options) {
        const headers = {};
        //
        // Access-Control-Allow-Headers
        const allowHeaders = options.allowHeaders || cors_1.Cors.DEFAULT_HEADERS;
        headers['Access-Control-Allow-Headers'] = `'${allowHeaders.join(',')}'`;
        //
        // Access-Control-Allow-Origin
        if (options.allowOrigins.length === 0) {
            throw new Error('allowOrigins must contain at least one origin');
        }
        if (options.allowOrigins.includes('*') && options.allowOrigins.length > 1) {
            throw new Error(`Invalid "allowOrigins" - cannot mix "*" with specific origins: ${options.allowOrigins.join(',')}`);
        }
        // we use the first origin here and if there are more origins in the list, we
        // will match against them in the response velocity template
        const initialOrigin = options.allowOrigins[0];
        headers['Access-Control-Allow-Origin'] = `'${initialOrigin}'`;
        // the "Vary" header is required if we allow a specific origin
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin#CORS_and_caching
        if (initialOrigin !== '*') {
            headers.Vary = '\'Origin\'';
        }
        //
        // Access-Control-Allow-Methods
        let allowMethods = options.allowMethods || cors_1.Cors.ALL_METHODS;
        if (allowMethods.includes('ANY')) {
            if (allowMethods.length > 1) {
                throw new Error(`ANY cannot be used with any other method. Received: ${allowMethods.join(',')}`);
            }
            allowMethods = cors_1.Cors.ALL_METHODS;
        }
        headers['Access-Control-Allow-Methods'] = `'${allowMethods.join(',')}'`;
        //
        // Access-Control-Allow-Credentials
        if (options.allowCredentials) {
            headers['Access-Control-Allow-Credentials'] = '\'true\'';
        }
        //
        // Access-Control-Max-Age
        let maxAgeSeconds;
        if (options.maxAge && options.disableCache) {
            throw new Error('The options "maxAge" and "disableCache" are mutually exclusive');
        }
        if (options.maxAge) {
            maxAgeSeconds = options.maxAge.toSeconds();
        }
        if (options.disableCache) {
            maxAgeSeconds = -1;
        }
        if (maxAgeSeconds) {
            headers['Access-Control-Max-Age'] = `'${maxAgeSeconds}'`;
        }
        //
        // Access-Control-Expose-Headers
        //
        if (options.exposeHeaders) {
            headers['Access-Control-Expose-Headers'] = `'${options.exposeHeaders.join(',')}'`;
        }
        //
        // statusCode
        const statusCode = options.statusCode !== undefined ? options.statusCode : 204;
        //
        // prepare responseParams
        const integrationResponseParams = {};
        const methodReponseParams = {};
        for (const [name, value] of Object.entries(headers)) {
            const key = `method.response.header.${name}`;
            integrationResponseParams[key] = value;
            methodReponseParams[key] = true;
        }
        return this.addMethod('OPTIONS', new integrations_1.MockIntegration({
            requestTemplates: { 'application/json': '{ statusCode: 200 }' },
            integrationResponses: [
                { statusCode: `${statusCode}`, responseParameters: integrationResponseParams, responseTemplates: renderResponseTemplate() },
            ],
        }), {
            methodResponses: [
                { statusCode: `${statusCode}`, responseParameters: methodReponseParams },
            ],
        });
        // renders the response template to match all possible origins (if we have more than one)
        function renderResponseTemplate() {
            const origins = options.allowOrigins.slice(1);
            if (origins.length === 0) {
                return undefined;
            }
            const template = new Array();
            template.push('#set($origin = $input.params("Origin"))');
            template.push('#if($origin == "") #set($origin = $input.params("origin")) #end');
            const condition = origins.map(o => `$origin.matches("${o}")`).join(' || ');
            template.push(`#if(${condition})`);
            template.push('  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)');
            template.push('#end');
            return {
                'application/json': template.join('\n'),
            };
        }
    }
    getResource(pathPart) {
        return this.children[pathPart];
    }
    /**
     * @internal
     */
    _trackChild(pathPart, resource) {
        this.children[pathPart] = resource;
    }
    resourceForPath(path) {
        if (!path) {
            return this;
        }
        if (path.startsWith('/')) {
            if (this.path !== '/') {
                throw new Error(`Path may start with "/" only for the resource, but we are at: ${this.path}`);
            }
            // trim trailing "/"
            return this.resourceForPath(path.substr(1));
        }
        const parts = path.split('/');
        const next = parts.shift();
        if (!next || next === '') {
            throw new Error('resourceForPath cannot be called with an empty path');
        }
        let resource = this.getResource(next);
        if (!resource) {
            resource = this.addResource(next);
        }
        return resource.resourceForPath(parts.join('/'));
    }
    get url() {
        return this.restApi.urlForPath(this.path);
    }
}
exports.ResourceBase = ResourceBase;
class Resource extends ResourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        validateResourcePathPart(props.pathPart);
        this.parentResource = props.parent;
        if (props.parent instanceof ResourceBase) {
            props.parent._trackChild(props.pathPart, this);
        }
        const resourceProps = {
            restApiId: props.parent.restApi.restApiId,
            parentId: props.parent.resourceId,
            pathPart: props.pathPart,
        };
        const resource = new apigateway_generated_1.CfnResource(this, 'Resource', resourceProps);
        this.resourceId = resource.ref;
        this.restApi = props.parent.restApi;
        // render resource path (special case for root)
        this.path = props.parent.path;
        if (!this.path.endsWith('/')) {
            this.path += '/';
        }
        this.path += props.pathPart;
        const deployment = props.parent.restApi.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ resource: resourceProps });
        }
        // setup defaults based on properties and inherit from parent. method defaults
        // are inherited per property, so children can override piecemeal.
        this.defaultIntegration = props.defaultIntegration || props.parent.defaultIntegration;
        this.defaultMethodOptions = {
            ...props.parent.defaultMethodOptions,
            ...props.defaultMethodOptions,
        };
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions || props.parent.defaultCorsPreflightOptions;
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
}
exports.Resource = Resource;
/**
 * Defines a {proxy+} greedy resource and an ANY method on a route.
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-set-up-simple-proxy.html
 */
class ProxyResource extends Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            parent: props.parent,
            pathPart: '{proxy+}',
            defaultIntegration: props.defaultIntegration,
            defaultMethodOptions: props.defaultMethodOptions,
        });
        const anyMethod = props.anyMethod !== undefined ? props.anyMethod : true;
        if (anyMethod) {
            this.anyMethod = this.addMethod('ANY');
        }
    }
    addMethod(httpMethod, integration, options) {
        // In case this proxy is mounted under the root, also add this method to
        // the root so that empty paths are proxied as well.
        if (this.parentResource && this.parentResource.path === '/') {
            // skip if the root resource already has this method defined
            if (!(this.parentResource.node.tryFindChild(httpMethod) instanceof method_1.Method)) {
                this.parentResource.addMethod(httpMethod, integration, options);
            }
        }
        return super.addMethod(httpMethod, integration, options);
    }
}
exports.ProxyResource = ProxyResource;
function validateResourcePathPart(part) {
    // strip {} which indicate this is a parameter
    if (part.startsWith('{') && part.endsWith('}')) {
        part = part.substr(1, part.length - 2);
        // proxy resources are allowed to end with a '+'
        if (part.endsWith('+')) {
            part = part.substr(0, part.length - 1);
        }
    }
    if (!/^[a-zA-Z0-9\.\_\-]+$/.test(part)) {
        throw new Error(`Resource's path part only allow [a-zA-Z0-9._-], an optional trailing '+'
      and curly braces at the beginning and the end: ${part}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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