"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Base class for backend integrations for an API Gateway method.
 *
 * Use one of the concrete classes such as `MockIntegration`, `AwsIntegration`, `LambdaIntegration`
 * or implement on your own by specifying the set of props.
 */
class Integration {
    constructor(props) {
        this.props = props;
    }
    /**
     * Allows `Method` to access the integration props.
     *
     * @internal
     */
    get _props() {
        return this.props;
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method
     * being integrated, access the REST API object, method ARNs, etc.
     */
    bind(_method) {
        return;
    }
}
exports.Integration = Integration;
var ContentHandling;
(function (ContentHandling) {
    /**
     * Converts a request payload from a base64-encoded string to a binary blob.
     */
    ContentHandling["CONVERT_TO_BINARY"] = "CONVERT_TO_BINARY";
    /**
     * Converts a request payload from a binary blob to a base64-encoded string.
     */
    ContentHandling["CONVERT_TO_TEXT"] = "CONVERT_TO_TEXT";
})(ContentHandling = exports.ContentHandling || (exports.ContentHandling = {}));
var IntegrationType;
(function (IntegrationType) {
    /**
     * For integrating the API method request with an AWS service action,
     * including the Lambda function-invoking action. With the Lambda
     * function-invoking action, this is referred to as the Lambda custom
     * integration. With any other AWS service action, this is known as AWS
     * integration.
     */
    IntegrationType["AWS"] = "AWS";
    /**
     * For integrating the API method request with the Lambda function-invoking
     * action with the client request passed through as-is. This integration is
     * also referred to as the Lambda proxy integration
     */
    IntegrationType["AWS_PROXY"] = "AWS_PROXY";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC. This integration is also referred to
     * as the HTTP custom integration.
     */
    IntegrationType["HTTP"] = "HTTP";
    /**
     * For integrating the API method request with an HTTP endpoint, including a
     * private HTTP endpoint within a VPC, with the client request passed
     * through as-is. This is also referred to as the HTTP proxy integration
     */
    IntegrationType["HTTP_PROXY"] = "HTTP_PROXY";
    /**
     * For integrating the API method request with API Gateway as a "loop-back"
     * endpoint without invoking any backend.
     */
    IntegrationType["MOCK"] = "MOCK";
})(IntegrationType = exports.IntegrationType || (exports.IntegrationType = {}));
var PassthroughBehavior;
(function (PassthroughBehavior) {
    /**
     * Passes the request body for unmapped content types through to the
     * integration back end without transformation.
     */
    PassthroughBehavior["WHEN_NO_MATCH"] = "WHEN_NO_MATCH";
    /**
     * Rejects unmapped content types with an HTTP 415 'Unsupported Media Type'
     * response
     */
    PassthroughBehavior["NEVER"] = "NEVER";
    /**
     * Allows pass-through when the integration has NO content types mapped to
     * templates. However if there is at least one content type defined,
     * unmapped content types will be rejected with the same 415 response.
     */
    PassthroughBehavior["WHEN_NO_TEMPLATES"] = "WHEN_NO_TEMPLATES";
})(PassthroughBehavior = exports.PassthroughBehavior || (exports.PassthroughBehavior = {}));
var ConnectionType;
(function (ConnectionType) {
    /**
     * For connections through the public routable internet
     */
    ConnectionType["INTERNET"] = "INTERNET";
    /**
     * For private connections between API Gateway and a network load balancer in a VPC
     */
    ConnectionType["VPC_LINK"] = "VPC_LINK";
})(ConnectionType = exports.ConnectionType || (exports.ConnectionType = {}));
//# sourceMappingURL=data:application/json;base64,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